;******************************************************************************
; File Name    	: keyer12F675-10.asm
; Version      	: 1.0
; Title:       	: Simple CW keyer
; Author		: F8EOZ Bernard DECAESTECKER http://f8eoz.com
; MCU			: Microchip PIC 12F675 Microcontroller
; Compiler     	: Microchip MPASM Assembler V5.51
; IDE          	: Microchip MPLAB IDE v8.92
; Programmer   	: Home made
; Last Updated 	: 01 July 2015
; *******************************************************************
; Hardware:     
;  Tested with PIC12F675 at 4Mhz  internal RC oscillator.       
;  Buzzer output is GP5, Pin 2 with transistor buffer.  
;  Keyed TX output is GP4, Pin 3 with transistor buffer.  
;  Dit paddle input is GP2, Pin 5, with external pullup.        
;  Dah paddle input is GP1, Pin 6, with external pullup.        
;  Keypad ADC input is GP0, Pin 7, with external pullup.
;  Reset is GP3	      
; *******************************************************************
; Description: 
; *******************************************************************
; Iambic or Straight keyer
; Keypad 9 functions:
;	3 prerecorded messages
;	Iambic mode A/B
;	ON/OFF side tone buzzer
;	Speed control +/-
;	Tune Pulse or Carrier
; Output side tone buzzer
; Output Tx connectivity
; Reset button
; *******************************************************************
	TITLE "keyer12F675-10.asm"
	LIST P=12F675, r=dec ;Sets the radix for data expressions. The default radix is hex. 
	ERRORLEVEL -302, -305	
	;errorlevel -302: Turn off banking message known tested (good) code
	;errorlevel +302: Enable banking message untested code

#include <p12F675.inc>

	__CONFIG   _CP_OFF & _CPD_OFF & _BODEN_OFF & _MCLRE_ON & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_CLKOUT  

;-----------------------------------------------------------------------------
;	Release
#define	V1_0
;-----------------------------------------------------------------------------
;	TEST and DEBUG
;	Comment out the next line [;#define TEST] if no test and debug
;#define	__TEST	; TEST and DEBUG mode
	ifdef	__TEST
		MESSG	"TEST and DEBUG = __TEST"
	endif
;#define	__TRON			; Buzzer TRACE ON (old programmer mania!)
;#define	__ADRES_TRON	; ADC result TRACE ON 
	ifdef	__TRON
		MESSG	"TEST and DEBUG = __TRON"
	endif
	ifdef	__ADRES_TRON
		MESSG	"TEST and DEBUG = __ADRES_TRON"
	endif
;-----------------------------------------------------------------------------
;	Programm and Data memory
#define RESET_VECTOR		0x0000	; Address of RESET Vector
#define ISR_VECTOR			0x0004	; Address of Interrupt Vector
#define RAM_DATA			0x0020	; 64 General Purpose registers (SRAM)
#define MSG_DATA			0x03d0
#define OSCCAL_FACTORY_ADR 	0x03FF
#define EEPROM_DATA			0x2100	; EEPROM address 
#define EEPROM_DATA_MAX		0x7F	; EEPROM 128 bytes 

;-----------------------------------------------------------------------------
; Input Output Port
;       GP0 - Analog input from keypad
;       GP1 - dah paddle
;       GP2 - dit paddle
;       GP3 - Reset button between ground and MCLR.  MCLR tied high via 10k resister.
;       GP4 - Drive Tx
;       GP5 - Drive Buzzer 
#define AN_KPAD 	GPIO,GP0	; (AN1) keypad
#define GP_DAH 		GP1			; (GP1) DAH paddle
#define GP_DIT 		GP2			; (GP2) DIT paddle
#define GP_TX 		GP4			; Tx output
#define GP_BUZ 		GP5			; Buzzer output

;----- UCON Bits -----------------------------------------------------
IAMBIC_KEY 	EQU 0x00	; StraightKey=0 Iambic= 1
BUZE	 	EQU 0x01	; = 1 if side tone buzzer enable
TXF 		EQU 0x02	; Tx flag = 1 if message sending to Tx
MSGF		EQU 0x03	; msg flag = 1 if send message running 
KPADF		EQU 0x04	; = 1 if keypad unlocked
MOD_B		EQU 0x05	; = 1 if keyer mode B, 0 mode A
BUZF		EQU 0x06	; buzzer flag = 1 if message sending to buzzer

;----- DEFAULT Options --------------------------------------------------
; Iambic Key + Buzzer on
#define UCON_DEFAULT (1<<IAMBIC_KEY) | (1<<BUZE) | (1<<BUZF)

;----- UCON1 Bits -----------------------------------------------------
; Compare Status Register
A_GT_B 		EQU 0x00	; = 1 if A>B
A_EQ_B 		EQU 0x01	; = 1 if A=B
A_LT_B 		EQU 0x02	; = 1 if A<B
; Paddle status
BOTHPADF	EQU 0x04	; = 1 if both paddles squeezed
IAMBIC_SW 	EQU 0x05	; Toggle iambic switch: 1=DIT	0=DAH

;-----------------------------------------------------------------------------
; MORSE
;-----------------------------------------------------------------------------
; Spacing and timing

; To standardize the International Code Transmission Speed, 
; the 5-letter word PARIS or CODEX is used to establish the number of words-per-minute unit 
; For example, if the word PARIS was sent 5 times in a minute, 
; the transmission speed would be 5-words-per-minute or WPM.
; The following relationships exist between the elements of the code (dits and dahs), 
; the characters (letters) and the words:
; The DIT is the Basic UNIT of Length.
; The DAH is equal in length to three DITS.
; The space between the DITS and DAHS within a character (letter) is equal to one DIT.
; The space between characters (letters) in a word is equal to three DITS.
; The space between words is equal to seven DITS.
; The following formula calculates the dot period in milliseconds  from
; the Morse code speed in words per minute:
; dot period = ( 1200 / speed )

; This formula arises from the use of the word PARIS as a standard word
; for calibrating Morse code speed.  PARIS is 50 units long when sent in
; Morse code. Analysis of English plain-text indicates that the average
; word is 50 units, including spaces.

;---------------------------------------------------------------------- 	
; Morse constant
#define X_TIME_5US 199		; time multiplier 	
#define DU 60 ;dot unit CODEX= 7 dots + (10 sign-gap * 1 dot) + (8 dashes * 3 dots) + (4 code-gap * 3 dots) + (1 word-gap * 7 dots)
;#define DU 50 ; dot unit PARIS= 10 dots + (9 sign-gap * 1 dot) + (4 dashes * 3 dots) + (4 code-gap * 3 dots) + (1 word-gap * 7 dots)
#define WG 7 ;  word-gap in dot unit

;---------------------------------------------------------------------- 	
; Morse code
;
; Letters A, B, C... Z = 1 to 4 bit coded character 
; xxxx: high nibble = Morse code; 1=DAH, 0=DIT,...
; yyyy: low nibble = 4 bit Morse code Frame vector. 1=DAH or DIT, 0=end
;             xxxxyyyy
cA EQU   	b'01001100'    ; A = .- 	(2 bit)
cB EQU   	b'10001111'    ; B = -... 	(4 bit)
cC EQU   	b'10101111'    ; C = -.-. 	(4 bit)
cD EQU    	b'10001110'    ; D = -.. 	(3 bit)
cE EQU    	b'00001000'    ; E = . 		(1 bit)
cF EQU    	b'00101111'    ; F = ..-.	(4 bit)
cG EQU   	b'11001110'    ; G = --.  	(3 bit)
cH EQU    	b'00001111'    ; H = ....	(4 bit)
cI EQU    	b'00001100'    ; I = ..		(2 bit)
cJ EQU    	b'01111111'    ; J = .---	(4 bit)
cK EQU    	b'10101110'    ; K = -.-  	(3 bit)
cL EQU    	b'01001111'    ; L = .-.. 	(4 bit)
cM EQU   	b'11001100'    ; M = --   	(2 bit)
cN EQU   	b'10001100'    ; M = -.   	(2 bit)
cO EQU   	b'11101110'    ; O = ---  	(3 bit)
cP EQU    	b'01101111'    ; P = .--.	(4 bit)
cQ EQU   	b'11011111'    ; Q = --.- 	(4 bit)
cR EQU    	b'01001110'    ; R = -.. 	(3 bit)
cS EQU   	b'00001110'    ; S = ...  	(3 bit)
cT EQU    	b'10001000'    ; T = - 		(1 bit)
cU EQU    	b'00101110'    ; U = ..-	(3 bit)
cV EQU    	b'00011111'    ; V = ...-	(4 bit)
cW EQU    	b'01101110'    ; W = .--  	(3 bit)
cX EQU   	b'10011111'    ; X = -..- 	(4 bit)
cY EQU   	b'10111111'    ; Y = -.-- 	(4 bit)
cZ EQU   	b'11001111'    ; Z = --.. 	(4 bit)

;---------------------------------------------------------------------- 	
; Morse code
; Numbers, Punctuation, Prosigns (Procedural signals)
; 8 bit coded character
; 3 bits b2...b0 = 010 = 5 bits format
; 3 bits b2...b0 = 011 = special format
; 2 bits b1...b0 = 01 = 6 bits format
; morse code; 1=DAH, 0=DIT,...
;
;
; Numbers 5 bit format
; 
;             xxxxx010 		-> Frame = 11111000
c1 	EQU  	b'01111010'    ; 1 = .---- 
c2 	EQU  	b'00111010'    ; 2 = .---- 
c3 	EQU  	b'00011010'    ; 3 = .---- 
c4 	EQU  	b'00001010'    ; 4 = .---- 
c5 	EQU  	b'00000010'    ; 5 = .---- 
c6 	EQU  	b'10000010'    ; 6 = -.... 
c7 	EQU  	b'11000010'    ; 7 = --... 
c8 	EQU  	b'11100010'    ; 8 = ---.. 
c9 	EQU  	b'11110010'    ; 9 = ----. 
c0 	EQU  	b'11111010'    ; 0 = ----- 

; Punctuation 5 bit format
;             		  xxxxx010		-> Frame = 11111000
cSlashBar 	EQU  	b'10010010'    ; / = -- 
cParentOpen	EQU  	b'10110010'    ; ( = ---
cAmpers		EQU  	b'01000010'    ; & = -
cEqual		EQU  	b'10001010'    ; = = -- 
cPlus 		EQU  	b'01010010'    ; + = .-.-.

; Punctuation 6 bit format
;             		  xxxxxx01		-> Frame = 11111100
cPer		EQU  	b'01010101'    ; . = ---
cComa		EQU  	b'11001101'    ; , = ----
cQM			EQU  	b'00110001'    ; ? = --
cQuot		EQU  	b'01111001'    ; ' = ----
cExcl		EQU  	b'10101101'    ; ! = ----
cParentClose	EQU  	b'10110101'    ; ) = ----
cCol		EQU  	b'11100001'    ; : = ---
cSCol		EQU  	b'10101001'    ; ; = --- 
cMinus		EQU  	b'10000101'    ; - = --
cUndS		EQU  	b'00110101'    ; _ = ---
cQuotM		EQU  	b'01001001'    ; " = --

; Prosigns special format
;             		  xxxxx011		-> code >127 Frame = 011
_WG 		EQU  	b'10000011'    ; WORD GAP = 2 dit time delay
_CS 		EQU  	b'10001011'    ; Alias for Call sign
_TUNEC	 	EQU  	b'10010011'    ; Tune mode constant carrier 
_TUNEP 		EQU  	b'10011011'    ; Tune mode 50% duty cycle dits
_ERR 		EQU  	b'10100011'    ; Error
_ERR8b		EQU  	b'00000000'    ; Error= ........ 8 bit format substitute
_SK 		EQU  	b'10101011'    ; SK
_SK6b 		EQU  	b'00010101'    ; SK = ...-.-  6 bit format substitute
_AR		 	EQU  	b'10110011'    ; AR
_AR5b	 	EQU  	b'01010010'    ; AR = .-.-.   5 bit format substitute
_BT		 	EQU  	b'10111011'    ; BT
_BT5b	 	EQU  	b'10001010'    ; BT = --	  5 bit format substitute
;
;----------------------------
; Header of message directive
_HOM		EQU     0x02
; End of message part
_EOP 		EQU		0x01
; End of message
_EOM 		EQU		0x00 

;-------
; KEYPAD 
;-------
; Definition
KPAD_00	EQU 0
KPAD_10	EQU 1
KPAD_20	EQU 2
KPAD_01	EQU 3
KPAD_11	EQU 4
KPAD_21	EQU 5
KPAD_02	EQU 6
KPAD_12	EQU 7
KPAD_22	EQU 8
KPAD_FF	EQU 0xFF ; no key
; Assign key 3x3 push button
; This allows one to change keypad layout without change algorithm
; My Layout:
; <Send0> <Spdup> <TuneC> (y2)
; <Send1> <Sdown> <TuneP> (y1)
; <Send2> <Buzz.> <ModAB> (y0)
;  (x2)	   (x1)	   (x0)
KPAD_SEND0		EQU KPAD_22
KPAD_SEND1		EQU KPAD_21
KPAD_SEND2	 	EQU KPAD_20
KPAD_SPEED_UP	EQU KPAD_12
KPAD_SLOW_DOWN 	EQU KPAD_11
KPAD_BUZ	 	EQU KPAD_10
KPAD_TUNEC		EQU KPAD_02
KPAD_TUNEP		EQU KPAD_01
KPAD_MOD		EQU KPAD_00

; Keypad maximum voltage threshold: analog to digital convert
; Ra = colomn resistor
; Rb = row resistor
; 3-by-3 keypad is a matrix x,y of resistor network divided into two row and column sections.
;
; Vref
;  |
; R(x,y)
;  |----->V(x,y)
; Rb
;  |
; Vss
;
; V(x,y) = Vref * ( Rb /(xRa + yRb + Rb)) (1)
; R(x,y) = xRa + yRb
; let Rb=nRa (n>2 to avoid duplicate value)
; equation (1) becomes V(x,y) = Vref * ( nRa /(xRa + nyRa + nRa)) = Vref * n /(x+ny+n) (2)
; let n= 3, equation (2) becomes V(x,y) = Vref * 3 /(x+3y+3) (3)

; Fitted after real analyze

#define VTOLPLUS 10 ; tolerance
#define VTOLMINUS 1 ; tolerance
; Row 0 (3069=1023*3)
#define KPAD_00_MAX (1023+VTOLPLUS)			;key 0,0 (maximum maximorum)
#define KPAD_00_MIN (1023-VTOLMINUS)		;key 0,0 
#define KPAD_10_MAX ((3069/4)+VTOLPLUS)		;key 1,0 
#define KPAD_10_MIN ((3069/4)-VTOLMINUS)	;key 1,0 
#define KPAD_20_MAX ((3069/5)+VTOLPLUS)		;key 2,0
#define KPAD_20_MIN ((3069/5)-VTOLMINUS)	;key 2,0 
; Row 1
#define KPAD_01_MAX ((1023/2)+VTOLPLUS)		;key 0,1
#define KPAD_01_MIN ((1023/2)-VTOLMINUS)	;key 0,1 
#define KPAD_11_MAX ((3069/7)+VTOLPLUS)		;key 1,1 
#define KPAD_11_MIN ((3069/7)-VTOLMINUS)	;key 1,1 
#define KPAD_21_MAX ((3069/8)+VTOLPLUS)		;key 2,1
#define KPAD_21_MIN ((3069/8)-VTOLMINUS)	;key 2,1 
; Row 2
#define KPAD_02_MAX ((1023/3)+VTOLPLUS)		;key 0,2
#define KPAD_02_MIN ((1023/3)-VTOLMINUS)	;key 0,2 
#define KPAD_12_MAX ((3069/10)+VTOLPLUS)	;key 1,2 
#define KPAD_12_MIN ((3069/10)-VTOLMINUS)	;key 1,2 
#define KPAD_22_MAX ((3069/11)+VTOLPLUS)	;key 2,2
#define KPAD_22_MIN ((3069/11)-VTOLMINUS)	;key 2,2 
; NO KEY
#define KPAD_FF_MAX VTOLPLUS				; no key 
#define KPAD_FF_MIN 0						; no key 

;---------------------------------------------------------------------- 	
	ifdef	__TEST
		#define	ISR_TMR0_OVF 1	; timer0 overflow 
	else  			
		#define	ISR_TMR0_OVF 11	; timer0 overflow 
	endif

; @4MHz instruction cycle = 1us
; TIMER1 prescaler = 1:8
; 16 bits TIMER1 = (256*256) = 65536
; 0.5 second = 500000us 
; init value = 65536 - (500000/8) = 3036 = 0xBDC
	ifdef	__TEST
		#define TMR1_INIT  65536 - 10 ; if test
		#define	TMR1_OVF_INIT 1	; timer0 overflow 
	else  			
		#define TMR1_INIT  (256*256) - (500000/8)
		#define TMR1_OVF_INIT  2 ; 2 overflow = 2*0.5 = 1s
	endif
 
;---------------------------------------------------------------------- 	
; ADC Config
#define ADC_CON b'10000001'
#define ADC_AN0 b'00000000'
#define ADC_AN1 b'00000100'
#define ADC_AN2 b'00001000'
#define ADC_AN3 b'00001100'
;         		  1-------  ADFM: 1=right justified result
;         		  -0------  VCFG: 0=Vdd is voltage reference
;         		  --xx----  not implemented
;         		  ----00--  00=select channel 00 (AN0=GP0)
;				      01 	= Channel 01 (AN1=GP1) 	
;				      10	= Channel 02 (AN2=GP2) 	
;					  11 	= Channel 03 (AN3=GP4)	
;         		  ------0-  GO/NOT DONE Status bit: 0=A/D conversion not started
;         		  -------1  ADON: Conversion Status bit 1=A/D converter module is operating

#define ADC_AN_KPAD ADC_AN0 ; assign Keypad to Channel 00

;-----------------------------------------------------------------------------
; PIC MACRO
;-----------------------------------------------------------------------------
BANK0	macro
		bcf	STATUS,RP0		
		endm

BANK1	macro
		bsf	STATUS,RP0		
		endm

;---------------------------------------------------------------------- 	
; DECODER MACRO
;---------------------------------------------------------------------- 	
; encoder: Input from a RAM message table
; input: _TAB_ADR_ = table address
; input: _DATA_ADR_ = data line address into table
; output WREG = relative address
SEND_SERVICE_MSG macro _TAB_ADR_, _DATA_ADR_
	bsf		UCON,BUZF		; Side tone buzzer on
	bcf		UCON,TXF		; unset flag Tx
	movlw	_DATA_ADR_ - _TAB_ADR_ - 1 - 3
	call 	Encoder_RAMinp	; Send text @WREG
	endm

;---------------------------------------------------------------------- 	
	cblock RAM_DATA
;---------------------------------------------------------------------- 	

	;--------------- 	
	; Register Files
	UCON:1		; 8 bits User Program Control Register
	UCON1:1		; 8 bits User Program Control Register 

	;-------------- 	
	; Counter Files 
	DelayCount:1		; delay counter

	;------------ 	
	; Timer Files
	TimeCount1:1        ; 8 bits counter
	TimeCount2:1        ; 8 bits counter
	WordPerMn:1			; 8 bits word per mn counter
	WordPerMnMax:1		; 8 bits maximum word per mn
	UnitPeriod:1     	; 8 bits current Unit Period counter
	SpeedPtr:1			; Speed scale table pointer

	;-------------- 	
	;Encoder Files
	WGapCount:1      	; 8 bits counter
	MorseCode:1			; 8 bits current Morse code
	ControlTape:1		; 8 bits current Morse code Frame vector
	BitFramePtr:1
	; Morse message Table
	TabPointer:1		; Table pointer 
	TabPointerSave:1	; Table pointer save for 2nd level
	;-------------------- 	
	; Key Interface Files
	GPIOcopy:1			; 8 bits current GPIO

	;----------------------- 	
	; Keypad Interface Files
	KeyPadXY:1     		; 8 bits current x,y key
	KeyPadXYPre:1     	; 8 bits previous x,y key
	KeyPadXYtemp:1     	; 8 bits temporary x,y key
	; ADC
	ADCVthPointer:1		; Table Voltage threshold pointer
	; Send message
	MsgN:1				; Message #
	BeaconTime:1		; Beacon Time of Message #

	;----------------------------- 	
	; Compare subroutine A?B Files
	CmpAL:1				; A LOW byte value
	CmpAH:1				; A HIGH byte value
	CmpBL:1				; B LOW byte value
	CmpBH:1				; B HIGH byte value
	
	;------------------------------- 	
	; math.sub Math subroutine Files
	; 16 bytes reseved
	AC1:1       		; 16-bit Accumulator a, lsb+1
	AC0:1       		; 16-bit Accumulator a, ls-byte

	;---------------- 	
	; Interrupt Files
	ISRwreg:1			; save WREG
	ISRstatus:1			; save STATUS
	TMR1ovf:1			; Timer1 overflow

	;------------- 	
	; EEPROM Files
	EEByteCounter:1	; EEPROM byte counter
	endc	; End of General Purpose registers                     
		
;---------------------------------------------------------------------- 	
; START ON  RESET  
;---------------------------------------------------------------------- 	

	org 	RESET_VECTOR 	; Reset vector
  	goto    Main			; Start program

	org		ISR_VECTOR      ; interrupt vector location
	goto	ISR

;---------------------------------------------------------------------- 	
; Timer: Speed Control: Speed scale Table
; input: SpeedFieldPtr = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
BIT_FRAME_TABLE
	movlw 	HIGH(BIT_FRAME); PCLATH = HIGH bit address
	movwf 	PCLATH 				; 		
	movf	BitFramePtr,W		; for PCL
	addwf	PCL ,F 				; Jump to character pointed to in W register
BIT_FRAME ; sort ascending by ASCII
#define ASCII_MIN " "
	DT  _WG ; Space ==>FIRST CODE 0x20 DON'T REMOVE
	DT  cExcl ; !
	DT  cQuotM ; "
	DT  _WG ; # not used
	DT  _WG ; $ not used
	DT  _WG ; % not used
	DT  cAmpers ; &
	DT  cQuot ; '
	DT  cParentOpen ; (
	DT  cParentClose ; )
	DT  _WG ; *
	DT  cPlus ; +
	DT  cComa ; ' not used
	DT  cMinus ; - (hyphen)
	DT  cPer ; . (dot)
	DT  cSlashBar ; /
	DT  c0
	DT  c1
	DT  c2
	DT  c3
	DT  c4
	DT  c5
	DT  c6
	DT  c7	
	DT  c8
	DT  c9
	DT  cCol ; :
	DT  cSCol ; ;
	DT  _WG ; < not used
	DT  cEqual ; =
	DT  _WG ; > not used
	DT  cQM ; ?
#define	CS_ALIAS "@"
	DT  _CS ;@
	DT	cA
	DT  cB
	DT  cC	
	DT  cD	
	DT	cE		
	DT	cF		
	DT  cG
	DT  cH
	DT  cI
	DT  cJ
	DT  cK
	DT  cL
	DT  cM
	DT  cN	
	DT  cO	
	DT  cP	
	DT  cQ	
	DT  cR	
	DT  cS	
	DT  cT	
	DT  cU	
	DT  cV
	DT  cW
	DT  cX	
	DT  cY	
	DT  cZ  ; ==>LAST CODE 0x5A DON'T REMOVE
#define ASCII_MAX "Z"
	IF ((HIGH ($)) != (HIGH (SPEED_SCALE)))
    	ERROR "BIT_FRAME CROSSES PAGE BOUNDARY!"
	ENDIF;

;---------------------------------------------------------------------- 	
; ADC Threshold Voltage Table
; ADC is 10 bits, and has a max. value of 0x3FF or 1023.
; With a 5 V reference, this gives a resolution of 5/1024
; or 4.88 mV.
; input: ADCVthPointer = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
ADC_Vth_TABLE
	movlw 	HIGH(ADC_Vth_KEYPAD); PCLATH = HIGH bit address
	movwf 	PCLATH 				; 		
	movf	ADCVthPointer,W		; for PCL
	addwf	PCL ,F ; Jump to character pointed to in W register
ADC_Vth_KEYPAD ; maximum voltage threshold for keypad: analog to digital convert 

	; real digit value saved into EEPROM for analyze (use __ADRES_TRON to get result)
	; 	(2)   (1)   (0)
	; 8---- 7---- 6---- (2)
	; 01 18 01 34 01 58 
	; 5---- 4---- 3---- (1)
	; 01 82 01 B8 02 01
	; 2---- 1---- 0---- (0)
	; 02 69 03 01 03 FF
    
	; -------xy: digit maximum voltage threshold 	: digit minimum voltage threshold 
	DT	KPAD_FF,(HIGH KPAD_FF_MAX),LOW (KPAD_FF_MAX),(HIGH KPAD_FF_MIN),LOW (KPAD_FF_MIN) 	
	DT	KPAD_22,(HIGH KPAD_22_MAX),LOW (KPAD_22_MAX),(HIGH KPAD_22_MIN),LOW (KPAD_22_MIN) 	
	DT	KPAD_12,(HIGH KPAD_12_MAX),LOW (KPAD_12_MAX),(HIGH KPAD_12_MIN),LOW (KPAD_12_MIN) 	
	DT	KPAD_02,(HIGH KPAD_02_MAX),LOW (KPAD_02_MAX),(HIGH KPAD_02_MIN),LOW (KPAD_02_MIN) 	
	DT	KPAD_21,(HIGH KPAD_21_MAX),LOW (KPAD_21_MAX),(HIGH KPAD_21_MIN),LOW (KPAD_21_MIN) 	
	DT	KPAD_11,(HIGH KPAD_11_MAX),LOW (KPAD_11_MAX),(HIGH KPAD_11_MIN),LOW (KPAD_11_MIN) 	
	DT	KPAD_01,(HIGH KPAD_01_MAX),LOW (KPAD_01_MAX),(HIGH KPAD_01_MIN),LOW (KPAD_01_MIN) 	
	DT	KPAD_20,(HIGH KPAD_20_MAX),LOW (KPAD_20_MAX),(HIGH KPAD_20_MIN),LOW (KPAD_20_MIN) 	
	DT	KPAD_10,(HIGH KPAD_10_MAX),LOW (KPAD_10_MAX),(HIGH KPAD_10_MIN),LOW (KPAD_10_MIN) 	
	DT	KPAD_00,(HIGH KPAD_00_MAX),LOW (KPAD_00_MAX),(HIGH KPAD_00_MIN),LOW (KPAD_00_MIN) 	; 5V
	; don't remove line below
	retlw 0xFF	; high value = end of table
	IF ((HIGH ($)) != (HIGH (ADC_Vth_KEYPAD)))
    	ERROR "ADC_Vth_KEYPAD CROSSES PAGE BOUNDARY!"
	ENDIF;
 
;---------------------------------------------------------------------- 	
; Timer: Speed Control: Speed scale Table
; input: SpeedFieldPtr = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
SPEED_SCALE_TABLE
	movlw 	HIGH(SPEED_SCALE); PCLATH = HIGH bit address
	movwf 	PCLATH 				; 		
	movf	SpeedPtr,W			; for PCL
	addwf	PCL ,F 				; Jump to character pointed to in W register
SPEED_SCALE ; speed and UnitPeriod
#define SPEED_MAX 50
#define SPEED_DEFAULT 12
;	   WPM:UnitPeriod--------
	DT	50,60000/((DU*50)-WG) ; 50 wpm max
	DT  49,60000/((DU*49)-WG)	
	DT  48,60000/((DU*48)-WG)	
	DT  47,60000/((DU*47)-WG)	
	DT	46,60000/((DU*46)-WG)		
	DT	45,60000/((DU*45)-WG)		
	DT  44,60000/((DU*44)-WG)
	DT  43,60000/((DU*43)-WG)
	DT  42,60000/((DU*42)-WG)
	DT  41,60000/((DU*41)-WG)
	DT  40,60000/((DU*40)-WG)
	DT  39,60000/((DU*39)-WG)
	DT  38,60000/((DU*38)-WG)
	DT  37,60000/((DU*37)-WG)	
	DT  36,60000/((DU*36)-WG)	
	DT  35,60000/((DU*35)-WG)	
	DT  34,60000/((DU*34)-WG)	
	DT  33,60000/((DU*33)-WG)	
	DT  32,60000/((DU*32)-WG)	
	DT  31,60000/((DU*31)-WG)	
	DT  30,60000/((DU*30)-WG)	
	DT  29,60000/((DU*29)-WG)
	DT  28,60000/((DU*28)-WG)
	DT  27,60000/((DU*27)-WG)	
	DT  26,60000/((DU*26)-WG)	
	DT  25,60000/((DU*25)-WG)
	DT  24,60000/((DU*24)-WG)	
	DT  23,60000/((DU*23)-WG)
	DT  22,60000/((DU*22)-WG)
	DT  21,60000/((DU*21)-WG)
	DT  20,60000/((DU*20)-WG)
	DT  19,60000/((DU*19)-WG)
	DT  18,60000/((DU*18)-WG)	
	DT  17,60000/((DU*17)-WG)
	DT  16,60000/((DU*16)-WG)
	DT  15,60000/((DU*15)-WG)	
	DT  14,60000/((DU*14)-WG)
	DT  13,60000/((DU*13)-WG)	
	DT  12,60000/((DU*12)-WG)
	DT  11,60000/((DU*11)-WG)
	DT  10,60000/((DU*10)-WG)
	DT  09,60000/((DU*09)-WG)	
	DT  08,60000/((DU*08)-WG)
	DT  07,60000/((DU*07)-WG)	
	DT  06,60000/((DU*06)-WG)
#define SPEED_MIN 50
	DT  05,60000/((DU*05)-WG) ; 5 wpm min
	; don't remove line below
	retlw 0	; low value = end of table
	IF ((HIGH ($)) != (HIGH (SPEED_SCALE)))
    	ERROR "SPEED_SCALE CROSSES PAGE BOUNDARY!"
	ENDIF;

;---------------------------------------------------------------------- 	
; Timer: Wait for one Unit Period long 	
; @4MHz one basic instruction duration = 1us 	
; @20WPM one Unit Period long UnitPeriod = 60 ms 	
; Routine wait for: 60 * (2us+(199 * 5us)+3us)= 60ms (+ 2us neglected) 	
; Input arg = UnitPeriod	
TimerUnitP 	
  	movfw	UnitPeriod		; Time Unit is argument (1us neglected)
  	movwf   TimeCount1      ; Counter for UnitPeriod * 1ms (1us neglected)	
; CALL for 1ms: set TimeCount1 before calling
; 1ms = 2us+(199 * 5us)+3us
TimerUnitP_1ms
  	movlw   X_TIME_5US			; 1 us
  	movwf   TimeCount2       	; 1 us
TimerUnitP_5us
  	nop                       	; 1 us to give 5 us inner loop 	
  	nop                       	; 1 us 	
  	decfsz  TimeCount2      	; 1 us if no skip
								; 2 us if skip	
  	goto    TimerUnitP_5us  	; 2 us 	
  
	decfsz  TimeCount1			; 1 us if no skip
								; 2 us if skip	
  	goto    TimerUnitP_1ms		; 2 us
  	return						; 2us neglected

;---------------------------------------------------------------------- 	
; Delay: Wait for (DelayCount * 5us) + 1us + 2us
; Input: WREG= factor DelayCount
Delay
  	movwf   DelayCount       	; 1 us
Delay_5us
  	nop                       	; 1 us to give 5 us inner loop 	
  	nop                       	; 1 us 	
  	decfsz  DelayCount      	; 1 us if no skip
								; 2 us if skip	
  	goto    Delay_5us		  	; 2 us 	
  	return						; 2us

;---------------------------------------------------------------------- 	
; Get speed from table 
; Input: WREG = Word Per Mn 
; Output: UnitPeriod
getUnitPeriod
	movwf	SpeedPtr
	addwf	SpeedPtr,W
	sublw	SPEED_SCALE - SPEED_SCALE_TABLE - 1 - 3 + SPEED_MAX + SPEED_MAX + 1	; W = value relative address 
	;movwf	SpeedItemPtr 		; Ptr to WPM
	;movwf	SpeedFieldPtr
	;incf	SpeedFieldPtr 		; Ptr to UnitP
	movwf	SpeedPtr			; Ptr to UnitPeriod
	call	SPEED_SCALE_TABLE
	movwf	UnitPeriod			; Load current Morse UnitPeriod
	return

;---------------------------------------------------------------------- 	
; Get EncCode from table 
; Input: WREG = ASCII 
; Output: WREG
getBitFrame
	movwf	BitFramePtr
	; is in table limit?
	sublw	ASCII_MAX ; W = h5A - W
	btfss	STATUS,C
	goto	getBitFrame_Out ; W<0: C=0
	movfw	BitFramePtr
	sublw	ASCII_MIN - 1 ; W = h19 - W
	btfsc	STATUS,C
	goto	getBitFrame_Out ; W>=0: C=1
	; yes: lookup table
	movlw	BIT_FRAME - BIT_FRAME_TABLE - 1 - 3 + ASCII_MIN	; W = value relative address "+"
	subwf	BitFramePtr,f
	call	BIT_FRAME_TABLE
	return
getBitFrame_Out
	movfw	BitFramePtr ; restore code
	return

;---------------------------------------------------------------------- 	
; Encoder: Input from EEPROM 
Encoder_EEPROMinp
	movlw	LOW(EE_MESSAGE) - 1	; msg start address
	BANK1
	movwf	EEADR	
	call	Encoder_EEPROMinp_GetHead
	xorlw	-1				; Check if error not found
	btfsc	STATUS, Z		; Z=1 returned if error not found
	goto	Encoder_EEPROMinp_Error
	BANK1
	movfw	EEADR			; hold address message
	BANK0
	movwf	TabPointer

;------------- 	
; Message loop
Encoder_EEPROMinp_Loop1
	btfss	UCON,MSGF		; is sending enabled?
	return	; no: abort sending >>>>>>>>
	movfw	TabPointer
	BANK1
	movwf	EEADR
	BANK0

;-------------- 	
; One code loop
Encoder_EEPROMinp_Loop2
	btfss	UCON,MSGF		; is sending enabled?
	return					; no: abort sending >>>>>>>>

	call	ReadEENextChar	; move code into WREG and set STATUS bit Z
	btfsc	STATUS, Z		; Check if end of message #: if Z=0: no
	goto	Encoder_EEPROMinp_EOM; yes: end ====>
	
	xorlw	CS_ALIAS		; Check if call sign (special bit format)
	btfsc	STATUS, Z		; Z=1 if = CS	 
	goto	Encoder_EEPROMinp_CS; yes, ====>
	xorlw	CS_ALIAS		; restore code 
	call	Encoder			; Encode
	goto	Encoder_EEPROMinp_Loop2

	; Enf Of Message
Encoder_EEPROMinp_EOM
	movlw	0xFF			; infinity loop?
	xorwf	BeaconTime,W
	btfsc	STATUS,Z
	goto 	Encoder_EEPROMinp_Loop1	; for ever loop message======>

	decfsz	BeaconTime	; Loop beacon time ===>
	goto	Encoder_EEPROMinp_Loop1

	return

;-------------------------------------------- 	
; Extend CS and insert CALL SIGN into message
Encoder_EEPROMinp_CS 	
	BANK1
	movfw 	EEADR		; Save current TabPointer
	BANK0
 	; New current pointer
	movwf	TabPointerSave
	movlw	LOW(EE_CALLSIGN) - 1 
	BANK1
	movwf 	EEADR		
	BANK0
Encoder_CallSign_CS_Loop
	btfss	UCON,MSGF		; is sending enabled?
	return					; no: abort sending >>>>>>>>
	call	ReadEENextChar	; move code into WREG and set STATUS bit Z
	btfsc	STATUS, Z		; Check if end of call sign: if Z=0: no
	goto	Encoder_CallSign_CS_EOM	; yes: end ====>
	;call	getBitFrame		;
	call	Encoder			; Encode
	goto	Encoder_CallSign_CS_Loop
Encoder_CallSign_CS_EOM
	movfw 	TabPointerSave	; Restore previous pointer TabPointer
	BANK1
	movwf	EEADR
	BANK0
	goto	Encoder_EEPROMinp_Loop2

;---------------------- 	
Encoder_EEPROMinp_Error
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, ERROR_HM_NOT_FOUND	; send service message
	return

;---------------------------------------------------------------------- 	
;Encoder: Input from EEPROM: get header of message 
; Input: MsgN = message #
; Output: WREG = 0 if found or -1 if not found
; Output: EEADR = address of last character of header (BeaconTime)
Encoder_EEPROMinp_GetHead
	; Message loop <====
	call	ReadEENextChar
	xorlw	_HOM				; Check if header message
	btfss	STATUS, Z		; Z=0: no
	retlw	-1				; error ====>
	call	ReadEENextChar
	xorwf	MsgN,W			; Check if message #
	btfsc	STATUS, Z		; Z=0: no
	goto	Encoder_EEPROMinp_GetHead_Found; yes: found ====>
	; Code of Message loop <====
	call	ReadEENextChar	; Loop until End of message
	btfss	STATUS, Z		; At end Z=1
	goto 	$-2				; no: next code loop ====>
	;
	goto 	Encoder_EEPROMinp_GetHead; next message loop ====>
Encoder_EEPROMinp_GetHead_Found
	call	ReadEENextChar
	movwf	BeaconTime		; Hold beacon time
	retlw	0

;---------------------------------------------------------------------- 	
;Encoder: Input from RAM table 
; Input: WREG = start address character
Encoder_RAMinp
	movwf	TabPointer		; Holds text address
Encoder_RAMinp_Loop

	call	SERVICE_MSG_TABLE
	andlw	0xFF				; Check if at end of text
	btfsc	STATUS, Z			; zero returned at end
	return						; ===>if 0 end of text            

	call	Encoder				; Send character
	incf	TabPointer,f		; Point to next character
	goto	Encoder_RAMinp_Loop

;	endEncoder_RAMinp

;---------------------------------------------------------------------- 	
;Encoder 
; WREG contains Morse code to encode	
Encoder 	
	call	getBitFrame		; Transcode character to BitFrame
  	movwf   MorseCode       ; hold Morse code
  	movwf   ControlTape  	; hold Bit frame
	;
	; switch case Frame key
	;
	; case: 5 BITS (as number)
	andlw	b'00000111'		; strip off b0...b3
	xorlw	b'00000010'		; Check if 5 bit format
	btfsc	STATUS, Z		; Z=1 if = 5 bit format	 
	goto	Encoder_5b		; yes, ====>
	;
	; case: 6 BITS (as punctuation)
	movfw	ControlTape
	andlw	b'00000111'		; strip off b0...b3
	xorlw	b'00000011'		; Check if >6 bit special format
	btfsc	STATUS, Z		; Z=1 if = >6 bit special format	 
	goto	Encoder_Spe		; yes, ====>
	; 
	; case: 6 BITS SPECIAL
	movfw	ControlTape
	andlw	b'00000011'		; strip off b0...b2
	xorlw	b'00000001'		; Check if 6 bit format
	btfsc	STATUS, Z		; Z=1 if = 6 bit format	 
	goto	Encoder_6b		; yes, ====>
	; 
	; case: 4 BITS (letter)
	; 4 bit format, frame is in low nibble
  	movfw   ControlTape     ; 
	andlw	b'00001111'		; strip off low nibble
 	movwf   ControlTape  	; hold low nibble frame
	swapf	ControlTape,f	; low nibble nibble -> high nibble nibble
	; 

; Send generic morse code
;------------------------ 	
Encoder_loop 	
	rlf		ControlTape,f; carry -> bit 0, bit 7 -> carry
	btfss	STATUS,C		; if carry send DIT or DAH
	goto	Encoder_GAP		; EXIT>>>>>if no carry end of code
	; Send DIT or DAH
	rlf		MorseCode, f	; bit 7 -> carry
	btfss	STATUS,C		; if carry send DAH
	goto	Encoder_DIT
Encoder_DAH		
	call 	Player_DAH
	goto	Encoder_loop	; -->loop Morse code
Encoder_DIT	
	call 	Player_DIT
	goto	Encoder_loop 	; -->loop Morse code
Encoder_GAP	
	call 	Player_GAP
	return				  	; EXIT>>>>>

; Send 5 BITS morse code (as number)
;----------------------------------- 	
Encoder_5b
	movlw	b'11111000'		; Frame = 5 bits
	movwf	ControlTape 	
	goto	Encoder_loop  	; -->loop Morse code

; Send 6 BITS morse code (as puntuation)
;--------------------------------------- 	
Encoder_6b
	movlw	b'11111100'		; Frame = 6 bits
	movwf	ControlTape 	
	goto	Encoder_loop  	; -->loop Morse code

; Send 8 BITS morse code (as puntuation)
;--------------------------------------- 	
Encoder_8b
	movlw	b'11111111'		; Frame = 8 bits
	movwf	ControlTape 	
	bcf		STATUS,C		; To close 8 bits frame
	goto	Encoder_loop  	; -->loop Morse code

; Send 6 BITS SPECIAL morse code
;------------------------------- 	
Encoder_Spe
	;
	; switch case Frame
	;
	; case: WORD GAP
	movfw	ControlTape
	xorlw	_WG				; Check if word gap (special bit format)
	btfsc	STATUS, Z		; Z=1 if = word gap	 
	goto	Encoder_WGap	; yes, ====>
	;
	; case: AR
	movfw	ControlTape
	xorlw	_AR				; Check if AR (special bit format)
	btfsc	STATUS, Z		; Z=1 if = 	 
	goto	Encoder_AR		; yes, ====>
	;
	; case: BT
	movfw	ControlTape
	xorlw	_BT				; Check if BT (special bit format)
	btfsc	STATUS, Z		; Z=1 if = 	 
	goto	Encoder_BT		; yes, ====>
	;
	; case: SK
	movfw	ControlTape
	xorlw	_SK				; Check if SK (special bit format)
	btfsc	STATUS, Z		; Z=1 if = 	 
	goto	Encoder_SK		; yes, ====>
	;
	; case: ERROR
	movfw	ControlTape
	xorlw	_ERR			; Check if error (special bit format)
	btfsc	STATUS, Z		; Z=1 if = ERROR	 
	goto	Encoder_Error	; yes, ====>
	;
	; case: TUNEC
	movfw	ControlTape
	xorlw	_TUNEC			; Check if tune c (special bit format)
	btfsc	STATUS, Z		; Z=1 if = TUNEC	 
	goto	Encoder_TuneC	; yes, ====>
	;
	; case: TUNEP
	movfw	ControlTape
	xorlw	_TUNEP			; Check if tune c (special bit format)
	btfsc	STATUS, Z		; Z=1 if = TUNEP	 
	goto	Encoder_TuneP	; yes, ====>
	
; UNDEFINED CODE
;--------------- 	
	return					; EXIT>>>>>>>>>>>>undefined code

; Send WORD GAP
;-------------- 	
Encoder_WGap
	; 1+2+2+2=7 between 1 word (1 wait was played below sign yet+2 below code)
    bcf    	GPIO,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX	    	; Keyer Output To Low
  	movlw   WG - (1+2+2)
	movwf	WGapCount
Encoder_WGap_loop
	call   	TimerUnitP		; Wait for one unitperiod long
 	decfsz  WGapCount      
   	goto   Encoder_WGap_loop 	
	return	; EXIT>>>>>>>>>>>>>>>>6 BITS SPECIAL CODE

; Tune mode constant carrier
;--------------------------- 	
Encoder_TuneC
	call	Player_CARRIER	
	return					; no: abort sending
							; EXIT>>>>>>>>>>>>>>>>6 BITS SPECIAL CODE

; Tune mode 50% duty cycle dits
; Pulse can be speeded up or slowed down by button before start tune mode
;------------------------------ 	
Encoder_TuneP
	call	Player_DIT	
	return					; no: abort sending
							; EXIT>>>>>>>>>>>>>>>>6 BITS SPECIAL CODE
 
; _AR
;---- 	
Encoder_AR
	movlw	_AR5b			; Subtitute bit Frame for 5 bits format
	movwf	MorseCode
	goto	Encoder_5b
 
; _BT
;---- 	
Encoder_BT
	movlw	_BT5b			; Subtitute bit Frame for 5 bits format
	movwf	MorseCode
	goto	Encoder_5b
 
; _SK
;---- 	
Encoder_SK
	movlw	_SK6b			; Subtitute bit Frame for 6 bits format
	movwf	MorseCode
	goto	Encoder_6b
 
; Error 8 dots
;------------- 	
Encoder_Error
	movlw	_ERR8b	; Subtitute bit Frame for 8 bits format
	movwf	MorseCode
	goto	Encoder_8b

;	endEncoder

;------------------------------------------------------------ 	
; CW KEYER: Player 
;------------------------------------------------------------ 	
; Play one DOT or DAH related to TXF and BUZF flags
;------------------------------------------------------------ 	
; 					! TXF 				! NOT TXF			!
;------------------------------------------------------------ 	
; 					! BUZF 	! NOT BUZF	! BUZF 	 ! NOT BUZF	!
;------------------------------------------------------------ 	
; SEND TO TX		! Yes	! Yes		! No	 ! No		!
; SEND TO BUZZER 	! Yes	! No (1)	! Yes (2)! Yes (2)	!
;------------------------------------------------------------ 
; (1) Buzzer side tone off	
; (2) Only for service message	
;------------------------------------------------------------ 
Player_DIT	
	btfsc	UCON,TXF		; Tx Output?
	bsf		GPIO,GP_TX    
	btfsc	UCON,BUZF		; Buzzer Output?
	bsf    	GPIO,GP_BUZ			
    call   	TimerUnitP		; Wait for one Unit Period long
    bcf    	GPIO,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX	    	; Keyer Output To Low
    call   	TimerUnitP		; Wait for one Unit Period long
	return
Player_DAH		
	btfsc	UCON,TXF		; Tx Output To High?
	bsf		GPIO,GP_TX    
	btfsc	UCON,BUZF		; Buzzer Output To High?
	bsf    	GPIO,GP_BUZ			
    call   	TimerUnitP		; Wait for one Unit Period long
    call   	TimerUnitP		; Wait for one Unit Period long
    call   	TimerUnitP		; Wait for one Unit Period long
    bcf    	GPIO,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX			; Keyer Output To Low
    call   	TimerUnitP		; Wait for one Unit Period long
	return
Player_GAP		
    ; 1+2 UnitPeriod between 2 codes (one Unit Period below one sign)
	call   	TimerUnitP		; Wait for one Unit Period long
    call   	TimerUnitP		; Wait for one Unit Period long
	return
Player_CARRIER ; special for tune mode		
	btfsc	UCON,TXF		; Tx Output?
	bsf		GPIO,GP_TX    
	btfsc	UCON,BUZF		; Buzzer Output?
	bsf    	GPIO,GP_BUZ			
	return

; End of Player
;---------------------------------------------------------------------- 	
;---------------------------------------------------------------------- 	
;---------------------------------------------------------------------- 	

;-------------------------------------------------------- 	
; 16 bit Compare A?B
; Input: CmpAH HIGH byte value A, CmpAL LOW byte value A, 
;		 CmpBH HIGH byte value B, CmpBL LOW byte value B
;		 Value register unchanged by routine
; Output: A_GT_B bit: 1 if A > B else 0
; Output: A_EQ_B bit: 1 if A = B else 0
; Output: A_LT_B bit: 1 if A < B else 0
;---------------------------------------------
; HIGH A > HIGH B 					!  A > B !
; HIGH A < HIGH B 					!  A < B !
; HIGH A = HIGH B AND LOW A > LOW B !  A > B !
; HIGH A = HIGH B AND LOW A = LOW B !  A = B !
; HIGH A = HIGH B AND LOW A < LOW B !  A < B !
;---------------------------------------------
CMP_A_B
	bsf		UCON1,A_GT_B		; init a > b
	bcf		UCON1,A_EQ_B		; init a not = b
	bcf		UCON1,A_LT_B		; init a not < b
	;
	; HIGH A > HIGH B ?
	movf	CmpAH,W
	subwf	CmpBH,W 	; W=CmpBH - CmpAH C=0 si CmpAH>CmpBH
    btfss   STATUS,C 	; CmpAH<=CmpBH: C=1
	return				; EXIT>>>>> A > B
	;
	; HIGH A = HIGH B ?
   	btfss   STATUS,Z 	
	goto	CMP_A_B_LT ; A < B
	;
	; CASE: HIGH A = HIGH B
	; 	lOW A > LOW B ?
	movf	CmpAL,W
	subwf	CmpBL,W 	; W=CmpBL - CmpAL C=0 si CmpAL>CmpBL
    btfss   STATUS,C 	; CmpAL<=CmpBL: C=1
	return				; EXIT>>>>>  A > B
	;
	; CASE: HIGH A = HIGH B
	; 	lOW A = LOW B ?
  	btfss   STATUS,Z 	
	goto	CMP_A_B_LT
	;
	; CASE: HIGH A = HIGH B AND	lOW A = LOW B
	bsf		UCON1,A_EQ_B		
 	bcf		UCON1,A_GT_B
	return				; EXIT>>>>> A = B
	;
	; CASE: A < B
CMP_A_B_LT
 	bcf		UCON1,A_GT_B
 	bsf		UCON1,A_LT_B
	return	; EXIT>>>>> a<b

;---------------------------------------------------------------------- 	
; KPAD_GetADCvalue
; Convert analog voltage level to 10 bits number
; Input: WREG=AD config with channel number to read
; Output: 10 bit ADC value is read from the pin and returned justified right.
; Please refer data sheet Page 41 (Section 7.1.4) for the AD Conversion
; clock explanation.  The 10-BIT ADC in the PIC12F675 are request
; minimum Tad with 1.6uS, and the conversion time need 11 Tad, so the
; mininum conversion time is 1.6uS * 11Tad = 17.6uS.

KPAD_GetADCvalue
	movwf	ADCON0			; AD config channel ANx
KPAD_GetADCvalue_Go
	movlw	15 		; wait for Acquisition time 
	call	Delay	; 15*5us=75us
	; A/D conversion cycle in progress
    bsf		ADCON0,GO_NOT_DONE	; Init GO_NOT_DONE       
    btfsc	ADCON0,GO_NOT_DONE	;<----------------------    
    goto	$-1					; while GO_NOT_DONE----> 
    return

;------------------------------------- 	
; KPAD_Keystroke process
; Scan ADC KPAD channel until find key
; Execute command associated with key
KPAD_Keystroke

KPAD_GetADRES
	movlw	ADC_CON | ADC_AN_KPAD	; select channel Keypad
	call	KPAD_GetADCvalue	
	BANK1					; store ADC result into compare register A
    movf	ADRESL,W       	; 8 bits low
	BANK0
	movwf	CmpAL
	movf	ADRESH,W        ; 2 bits High	
	movwf	CmpAH
	
	; point to first ADC table item
	movlw	ADC_Vth_KEYPAD - ADC_Vth_TABLE - 1 -3 -3 	; W = value relative address - 3
	movwf	ADCVthPointer		; Hold pevious line address

KPAD_Lookup	; read one table item
	incf	ADCVthPointer		; skip min value of previous table item
	incf	ADCVthPointer			 
	incf	ADCVthPointer		; point to current table item
	call	ADC_Vth_TABLE		; lookup voltage scale to find key pressed
	movwf   KeyPadXYtemp       	; hold KeyPadXY byte

	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
 	movwf   CmpBH       		; hold MAX value high byte into compare register B
	
	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
   	movwf   CmpBL	       		; hold MAX value low byte into compare register B

 	call	CMP_A_B				; compare ADC : MAX value
	btfss	UCON1,A_LT_B		; is ADC value < MAX table value ?
	goto	KPAD_Lookup			; no: NEXT item------>

	; value is < MAX: then look for MIN value
	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
 	movwf   CmpBH       		; hold MIN value high byte into compare register B
	
	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
   	movwf   CmpBL	       		; hold MIN value low byte into compare register B

 	call	CMP_A_B				; compare ADC : MIN value
	btfsc	UCON1,A_EQ_B		; is ADC value = MIN table value
	goto	KPAD_GetKey_Found	; yes: Key found------>
	btfss	UCON1,A_GT_B		; is ADC value > MIN table value
	goto	KPAD_GetADRES		; no: value out of voltage span, read again------>

KPAD_GetKey_Found
 	movf	KeyPadXY,W			; Hold previous key
	movwf	KeyPadXYPre			
 	movf	KeyPadXYtemp,W		; Hold new key
	movwf	KeyPadXY			

;-------------------------------
; only for analyze and benchmark
KPAD_Report	
	ifdef __ADRES_TRON	
 		; Report ADRES value into EEPROM for analyse
		addwf	KeyPadXY,W		; pointer = 2*KeyPadXY
		btfsc	STATUS,C		; carry set if key 0xFF
		goto	KPAD_Report_End	; >>>no report
		sublw	LOW(EE_ADRESL0); pointer to LSB
		BANK1
		movwf	EEADR			; into EEADR
		movf	ADRESL,W		; ADC value LSB into EEDATA
		movwf	EEDATA			
		call	WriteEEPROM		; write one byte to EEPROM
		BANK1
		decf	EEADR,f			; pointer to MSB
		BANK0
		movf	ADRESH,W		; ADC value MSB into EEDATA
		BANK1
		movwf	EEDATA			
		call	WriteEEPROM		; write one byte to EEPROM
		BANK0
	endif
KPAD_Report_End
;--------------

	; switch case KeyPadXY
	;
	; case: NO KEY
	movfw	KeyPadXY
	xorlw	KPAD_FF				; Check if no key
	btfsc	STATUS, Z			; Z=0: no next key	 
	goto	KPAD_Keystroke_End	; Z=1: yes, end key ====>
	;
	; case: KEY = PREVIOUS KEY
	movfw	KeyPadXY
	xorwf	KeyPadXYPre,W		; Check if equal
	btfsc	STATUS, Z			; Z=0: no
	goto	KPAD_Keystroke_End	; end key ====>
	;
	; case: SEND0
	movfw	KeyPadXY
	xorlw	KPAD_SEND0		; Check if key Send0
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_Sendn	; yes, ====>
	;
	; case: SEND1
	movfw	KeyPadXY
	xorlw	KPAD_SEND1		; Check if key Send1
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_Sendn	; yes, ====>
	;
	; case: SEND2
	movfw	KeyPadXY
	xorlw	KPAD_SEND2		; Check if key Send2
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_Sendn	; yes, ====>
	;
	; case: BUZZER
	movfw	KeyPadXY
	xorlw	KPAD_BUZ		; Check if key Buz
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_Buz	; yes, ====>
	;
	; case: TUNEP
	movfw	KeyPadXY
	xorlw	KPAD_TUNEP		; Check if key Tune PULSE
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_TunP	; yes, ====>
	;
	; case: TUNEC
	movfw	KeyPadXY
	xorlw	KPAD_TUNEC		; Check if key Tune CARRIER
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_TunC	; yes, ====>
	;
	; case: MODEAB
	movfw	KeyPadXY
	xorlw	KPAD_MOD		; Check if key Mod
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_Mod	; yes, ====>
	;
	; case: SPEED UP
	movfw	KeyPadXY
	xorlw	KPAD_SPEED_UP	; Check if key Speed up
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_SpeedUp	; yes, ====>
	;
	; case: SLOW_DOWN
	movfw	KeyPadXY
	xorlw	KPAD_SLOW_DOWN	; Check if key Slow down
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_SlowDown	; yes, ====>
	;
KPAD_Keystroke_End ; End of keypad
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Send message #0,#1,#2 to Tx
;---------------------------- 	
KPAD_Keystroke_Sendn
	ifdef __TRON
		movlw	"S"
		call	Encoder
	endif

	btfss	UCON,BUZE		; If side tone buzzer disable
	bcf		UCON,BUZF		; unset flag buzzer
	bsf		UCON,TXF		; set flag Tx

	movfw	KeyPadXY ; Button assign = message #
	movwf	MsgN

	bsf		UCON,MSGF		; enable message sending
							; sending may be interrupted by DIT paddle
	bcf		INTCON,INTF
	bsf		INTCON,INTE		; enable interrupt by DIT paddle
	; SEND ON
	call 	Encoder_EEPROMinp

	bsf		UCON,BUZF		; set flag buzzer
	bcf		UCON,TXF		; unset flag Tx

	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Side Tone Buzzer ON/OFF
; Only for TX message
; Not for service message
;------------------------ 	
KPAD_Keystroke_Buz
	movlw  	(1 << BUZE)
    xorwf  	UCON,f  ; Toggle buzzer bit
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, BUZ_MSG	; send service message
	call 	SaveOption
	return

; Tune mode constant carrier
;--------------------------- 	
KPAD_Keystroke_TunC
	btfss	UCON,BUZE		; If side tone buzzer disable
	bcf		UCON,BUZF		; unset flag buzzer
	bsf		UCON,TXF		; set flag Tx

	bsf		UCON,MSGF		; enable message sending
							; sending may be interrupted by DIT paddle
	bcf		INTCON,INTF
	bsf		INTCON,INTE		; enable interrupt by DIT paddle
	; carrier ON
	movlw	_TUNEC
	call	Encoder
	
	btfsc	UCON,MSGF		; is sending enabled? <----
	goto	$-1				; yes: loop -------------->
	; carrier OFF
    bcf    	GPIO,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX			; Keyer Output To Low

	bcf		INTCON,INTE		; disable interrupt DIT paddle

	bcf		UCON,TXF		; unset flag Tx
	bsf		UCON,BUZF
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Tune mode pulse 50% duty cycle dits
;------------------------------------ 	
KPAD_Keystroke_TunP
	ifdef __TRON
		movlw	"P"
		call	Encoder
	endif

	btfss	UCON,BUZE		; If side tone buzzer disable
	bcf		UCON,BUZF		; unset flag buzzer
	bsf		UCON,TXF		; set flag Tx

	bsf		UCON,MSGF		; enable message sending
							; sending may be interrupted by DIT paddle
	bcf		INTCON,INTF
	bsf		INTCON,INTE		; enable interrupt by DIT paddle
	; Pulse ON
	movlw	_TUNEP			; <-------------------------
	call	Encoder
	btfsc	UCON,MSGF		; is sending enabled?
	goto	$-3				; yes: loop  -------------->
	; Pulse OFF

	bcf		INTCON,INTE		; disable interrupt DIT paddle

	bcf		UCON,TXF		; unset flag Tx
	bsf		UCON,BUZF
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Toggle mode A or B
;------------------- 	
KPAD_Keystroke_Mod
	movlw  	(1 << MOD_B)
    xorwf  	UCON,f        	; Toggle mod bit
	call 	SaveOption
	btfss	UCON,MOD_B
	goto	KPAD_Keystroke_Mod_A
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, MOD_B_MSG	; send service message
	return
KPAD_Keystroke_Mod_A
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, MOD_A_MSG	; send service message
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Speed up
;--------- 	
KPAD_Keystroke_SpeedUp
	movf	WordPerMn,W		; Check if max value
	xorlw	SPEED_MAX			
	btfsc	STATUS, Z		; Z=1 returned at end
	return	; EXIT>>>>>end of speed scale table            

	incf	WordPerMn
	movfw	WordPerMn
	call	getUnitPeriod		; get UnitPeriod 
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, SPEED_UP_MSG	; send service message
	call 	SaveOption
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Slow down
;---------- 	
KPAD_Keystroke_SlowDown
	movf	WordPerMn,W		; Check if min value
	xorlw	SPEED_MIN			
	btfsc	STATUS, Z		; Z=1 returned at end
	return	; EXIT>>>>>end of speed scale table            

	decf	WordPerMn
	movfw	WordPerMn
	call	getUnitPeriod	; get UnitPeriod 
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, SLOW_DOWN_MSG	; send message
	call 	SaveOption
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

;---------------------------------------------------------------------- 	
; Automatic set type of key: iambic or straight
; Input: GPIO,GP_DAH	
; Output: UCON,IAMBIC_KEY flag	
; iambic key: if DAH not grounded
; straight key if DAH grounded
SetKeyType
	btfss	GPIO,GP_DAH				; iambic or straight key?
	goto	SetKeyType_S
	;
	bsf		UCON,IAMBIC_KEY 	
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, IAMBIC_MSG	; iambic message
	return	; >>>>>>>>>>>>
SetKeyType_S
	bcf		UCON,IAMBIC_KEY 	
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, STRAIGHT_MSG; straight message
	return

;---------------------------------------------------------------------- 	
; Set TIMER1 interrupt
; TIMER1 is settle for 1s wait
SetIntTIMER1
    
 	BANK0					
    btfsc  	UCON, KPADF 	; Is keypad locked?
	return					; yes: no more interrupt>>>>>>

	movlw  	TMR1_OVF_INIT  		; set max TMR1 overflow
	movwf	TMR1ovf
    movlw   LOW(TMR1_INIT)		; clear TRM1 refore reset flag
    movwf   TMR1L
    movlw   HIGH(TMR1_INIT)
    movwf   TMR1H
	
	bcf     PIR1, TMR1IF    	; clear TMR1 interrupt flag
    banksel PIE1
	bsf     PIE1, TMR1IE   		; enable TMR1 interrupts
	BANK0
	return

;------------------------- 	
; Load options from EEPROM									
;
LoadOption
	; Set last value of WPM
	movlw	LOW(EE_WPM)
	call	ReadEEChar
	movwf	WordPerMn		; into WordPerMn option
	
	; Set last value of User Program control Register
	movlw	LOW(EE_UCON)	
	call	ReadEEChar
	movwf	UCON			; into Program Control Register

	return

;---------------------------------------------------------------------- 	
; EEPROM GENERAL SUBROUTINE
;---------------------------------------------------------------------- 	
;----------------------------------------- 	
; Write EEPROM	
; Input EEDAT: data to put in EEPROM
; Input	EEADR: address of data into EEPROM
;
WriteEEPROM
	BANK1
	bcf EECON1,EEIF 	;disable interrupt write to EEPROM
	bsf EECON1,WREN 	;enable write to EEPROM
	movlw 0x55
	movwf EECON2
	movlw 0xaa
	movwf EECON2
	bsf EECON1,WR		; set WR = write EEPROM
WriteEEwait
	btfsc EECON1,WR		; loop <---------------
	goto $-1			; loop until WR = 0 -->
	bcf EECON1,WREN 	; disable write to EEPROM
	BANK0
	return 

;---------------------------------------------------------------------- 	
; Save options into EEPROM	
;
SaveOption
	movf	WordPerMn,W			; WPM into EEDATA
	BANK1
	movwf	EEDATA			
	movlw	LOW(EE_WPM)			; EEPROM WPM output address
	movwf	EEADR
	call 	WriteEEPROM

	BANK0
	movf	UCON,W				; User Flag into EEDATA
	BANK1
	movwf	EEDATA			
	movlw	LOW(EE_UCON)		; EEPROM User Flag output address
	movwf	EEADR
	call 	WriteEEPROM
	BANK0
	return 

;---------------------------------------------------------------------- 	
; Read character from EEPROM @address 
; Input: WREG = relative address
; Output: WREG= character
ReadEEChar
	BANK1
	movwf	EEADR	
	bsf		EECON1,RD		; read
	movf	EEDATA,W		; data ouput Z=1 if zero
	BANK0
	return

;---------------------------------------------------------------------- 	
; Read next character from EEPROM
; Input: EEADR = address of previous character
; Output: WREG = character @EEADR+1
; Output: EEADR + 1= address of current character
ReadEENextChar
	BANK1
	movlw	EEPROM_DATA_MAX		; Check if out of memory
	xorwf	EEADR,W				; Check if out of memory
	btfsc	STATUS, Z			; Z=1 returned if error not found
	goto	ReadEENextChar_Error
	incf	EEADR
	bsf		EECON1,RD		; read
	movf	EEDATA,W		; data ouput Z=1 if zero
	BANK0
	return
ReadEENextChar_Error ; Out memory protect
	BANK0
	retlw	_ERR ; return with STATUS Z bit below = 1
	return

;---------------------------------------------------------------------- 	
; PROGRAM ROOT 
;---------------------------------------------------------------------- 	
Main
	; Set up Oscillator calibration value
	call    OSCCAL_FACTORY_ADR	; retrieve factory calibration value
	banksel OSCCAL				; and save onto register
	movwf   OSCCAL      

    banksel GPIO
    movlw   b'00000000'	; all ports low level
    movwf   GPIO

	; set up I/O port direction
    banksel TRISIO
    movlw   b'00000111'
    ;         xx------  not implemented
    ;         --0-----  0=output, GP5, TX control
    ;         ---0----  0=output, GP4, Buzzer control
    ;         ----x---  not used, GP3, MCLR and reset
    ;         -----1--  1=input, GP2, DIT paddle input /INT
    ;         ------1-  1=input, GP1, DAH paddle input
    ;         -------1  1=input, GP0, keypad analog input
    movwf   TRISIO

	; set up option register
	banksel OPTION_REG
    movlw   b'11111111'
    ;         1-------  NOT GPPU: 1 = GPIO pull-ups are disabled
    ;         -1------  INTEDG: 0 = Interrupt on rising edge of GP2/INT pin
    ;         --1-----  T0CS: 1 = Transition on GP2/T0CKI pin0=output, GP4, Buzzer control
    ;         ---1----  T0SE: 1 = Increment on high-to-low transition on GP2/T0CKI pinnot used, GP3, MCLR and reset
    ;         ----1---  PSA: 1 = Prescaler is assigned to the WDT1=input, GP4, DAH paddle input
    ;         -----111  PS2:PS0: 111= 1:128 prescaler
  	movwf	OPTION_REG	

	; set up comparator
    banksel CMCON
    movlw   b'00000111'
    ;         x-------  not implemented
    ;         -0------  COUT: Comparator Output bit: 0 = VIN+ < VIN- When CINV = 0
    ;         --x-----  not implemented
    ;         ---0----  CINV: Comparator Output Inversion bit 0=non-inverted output
    ;         ----0---  CIS: Comparator Input Switch bit When CM2:CM0 = 110 or 101
    ;         -----111  CM2:CM0: Comparator Mode bits 111=comparator off
    movwf   CMCON

	; set up A/D converter
    banksel ANSEL
    movlw   b'00010001'
    ;         x-------  not implemented
    ;         -001----  ADCS A/D Conversion Clock: 001=Focs/8 Conversion Clock
    ;         -101----  ADCS A/D Conversion Clock: 001=Focs/16 Conversion Clock
    ;         ----0---  ANS3:ANS0: Analog Select bits 0=digital I/O, GP4, Tx output
    ;         -----0--  ANS3:ANS0: Analog Select bits 0=digital I/O, GP2, DIT paddle input
    ;         ------0-  ANS3:ANS0: Analog Select bits 0=digital I/O, GP1, DAH paddle input
    ;         -------1  ANS3:ANS0: Analog Select bits 1=analog  I/O, GP0, analog input
    movwf   ANSEL

	; set up A/D converter
	banksel ADCON0
    movlw   ADC_CON | ADC_AN_KPAD
    movwf   ADCON0

	; initialize interrupts
    banksel INTCON
    movlw   b'01000000'
    ;         0-------  0=interrupts disabled
    ;         -1------  0=enable peripheral interrupts
    ;         --0-----  0=disable TMR0 overflow interrupt
    ;         ---0----  0=disable GP2/INT external interrupt
    ;         ----0---  0=disable GPIO port change interrupt
    ;         -----0--  0=no on TMR0 overflow
    ;         ------0-  0=no GP2/INT external interrupt
    ;         -------0  0=no GPIO port change
    movwf   INTCON
    
    banksel PIE1 	; in bank 1   
    movlw   b'00000000'
    ;         0-------  0=disable EE write complete interrupt
    ;         -0------  0=disable A/D converter interrupt
    ;         --xx----  not implemented
    ;         ----0---  0=comparator interrupt disabled
    ;         -----xx-  not implemented
    ;         -------0  0=disable TMR1 overflow interrupt
    movwf   PIE1
        
    banksel PIR1	; in bank 0
    movlw   b'00000000'
    ;         0------- 0=no EE write complete
    ;         -0------ 0=no A/D conversion complete
    ;         --xx---- not implemented
    ;         ----0--- 0=no comparator interrupt
    ;         -----xx- not implemented
    ;         -------0 0=no TMR1 overflow
    movwf   PIR1

	; initialize TMR1
    banksel T1CON
    movlw  b'00110101'    
    ;        x-------  not implemented
    ;        -0------  0=gate disabled
    ;        --11----  01=1:8 prescaler
    ;        ----0---  0=LP oscillator is off
    ;        -----1--  1=external clock input not synchronized
    ;        ------0-  0=internal clock
    ;        -------1  1=enable timer
    movwf   T1CON
     
	BANK0

	clrf	UCON1				; All user flags 0
	movlw	KPAD_FF				; Key pad = no key pushed
	movwf	KeyPadXY
	movwf	KeyPadXYPre

	call	LoadOption			; Load options from EEPROM
	
	movfw	WordPerMn			
	call	getUnitPeriod		; get UnitPeriod

	BANK0

	clrf	TMR1L			; TMR1=0
   	clrf	TMR1H
	bsf     INTCON,GIE     	; enable all interrupts

	call	SetKeyType		; Iambic or Straight key

;----------------------------------------------------------
; TMR1IE overflow interrupts are disabled by any DIT or DAH 
; Keypad is locked by any DIT or DAH
Main_Loop
	btfss	UCON,IAMBIC_KEY	; 2 WAYS PROCESS: IAMBIC OR STRAIGHT KEY
	goto 	StraightKey

;----------------------------
; IAMBIC KEY MODE A OR MODE B
;----------------------------
IambicKey
	BANK0
	movfw	GPIO							; GPIO snapshot
	andlw	(1 << GP_DIT)|(1 << GP_DAH) 	; strip off paddle port
	movwf	GPIOcopy
	;
	; switch case GPIOcopy
	;
	; case: DIT KEY DOWN
	movfw	GPIOcopy
	xorlw	(0 << GP_DIT)|(1 << GP_DAH)	; Check if DIT down
	btfsc	STATUS, Z		; Z=0: no
	goto	IambicKey_DIT	; end key ---->
	;
	; case: DAH KEY DOWN
	movfw	GPIOcopy
	xorlw	(1 << GP_DIT)|(0 << GP_DAH)	; Check if DAH down
	btfsc	STATUS, Z		; Z=0: no
	goto	IambicKey_DAH	; end key ---->
	;
	; case: DIT and DAH SQUEEZE BOTH
	movfw	GPIOcopy
	xorlw	(0 << GP_DIT)|(0 << GP_DAH)	; Check if 2 keys down
	btfsc	STATUS, Z			; Z=0: no next key	 
	goto	IambicKey_BOTHPAD	; Z=1: yes, end key ---->
	;
	; case: NO DIT and NO DAH
	btfss	UCON1,BOTHPADF
	goto	KPAD_Interface				; no key down: scan keypad?=======>
	btfss	UCON,MOD_B
	goto	KPAD_Interface				; no key down: scan keypad?=======>
IambicKey_RELEASE
 	; Mode B keyer sends an additional element opposite 	
	; to the one being sent when the paddles are released.	
	movlw  	(1 << IAMBIC_SW)
    xorwf  	UCON1,f        	; Toggle iambic bit
	btfss	UCON1,IAMBIC_SW
	goto	IambicKey_DAH
IambicKey_DIT
	; yes! send one DIT	
	bcf 	UCON1,BOTHPADF 	; Both paddles released
IambicKey_DIT_Both
	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupt
	BANK0
	bsf		UCON1,IAMBIC_SW		; settle Iambic switch
	bcf		UCON, KPADF 		; Lock Keypad
	bsf		UCON,TXF			; set flag Tx
	call 	Player_DIT
	bcf		UCON,TXF			; unset flag Tx
	goto	IambicKey  			; loop for another input ---->
IambicKey_DAH
	bcf 	UCON1,BOTHPADF 		; Both paddles released
IambicKey_DAH_Both
 	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupt
	BANK0
	bcf		UCON1,IAMBIC_SW		; settle Iambic switch
    bcf		UCON, KPADF 		; Lock Keypad
	bsf		UCON,TXF			; set flag Tx
	call 	Player_DAH
	bcf		UCON,TXF			; unset flag Tx
	goto	IambicKey  			; Loop for another input ---->
IambicKey_BOTHPAD
	movlw  	(1 << IAMBIC_SW)
    xorwf  	UCON1,f        		; Toggle iambic bit
	bsf 	UCON1,BOTHPADF 		; Both paddles squeezed
	btfss	UCON1,IAMBIC_SW
	goto	IambicKey_DAH_Both	; ---->
	goto	IambicKey_DIT_Both  ; ---->

;-------------
; STRAIGHT KEY
;-------------
StraightKey
	btfsc	GPIO,GP_DIT
	goto	StraightKey_Up		; no key down: ---->
;	Straight key down
	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupt
	BANK0
    bcf		UCON, KPADF 		; Lock Keypad
	bsf    	GPIO,GP_TX	   		; Tx Output
	btfsc	UCON,BUZE			; Buzzer Output Enable?
	bsf    	GPIO,GP_BUZ			
	goto	StraightKey
StraightKey_Up
	bcf    	GPIO,GP_TX	    	; Tx Output
	bcf    	GPIO,GP_BUZ	    	; Buzzer Output
	goto	KPAD_Interface		; no key down: scan keypad? ====>

;-------------------------------- 	
; SCAN KEYPAD ON INTERRUPT TIMER1
;-------------------------------- 	
KPAD_Interface
    btfss  	UCON, KPADF 		; Is keypad unlocked?
	goto 	KPAD_Interface_TMR1	; no: set TIMER1 ------>
	
	;call	KPAD_GetKey			; yes! keypad ready to use
	call 	KPAD_Keystroke
	goto	Main_Loop			; Loop for another input ------>

KPAD_Interface_TMR1
	banksel PIE1
	btfss	PIE1, TMR1IE   		; Is TMR1 interrupt ready?
	call 	SetIntTIMER1		; no: get it ready
	BANK0						; yes: wait for interrupt
	goto	Main_Loop			; Loop for another input ------>

;------------------ 	
; INTERRUPT MANAGER
;------------------ 	
ISR
	;save register
	movwf	ISRwreg  		; WREG
	swapf	STATUS,w		; STATUS without change Z
	movwf	ISRstatus		; 
        
ISR_TMR1
 	; check for TMR1 overflow interrup switch
	; beware of bank! PIR1 is in bank0, PIE1 in bank1
	banksel PIE1
	btfss	PIE1,TMR1IE			; interrupt timer enable ?
	goto 	ISR_TMR1_END		; no: next interrupt

	; yes: interrupt timer running?
    BANK0
	btfss	PIR1,TMR1IF		
	goto 	ISR_TMR1_END		; no: end of interrupt

	; yes TMR1 overflow ?
    decfsz  TMR1ovf
	goto	ISR_TMR1_rollover	; no: counter rollover

    ; end of count
	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupts
    BANK0
	bcf     PIR1, TMR1IF    	; clear interrupt flag
    bsf     UCON, KPADF  		; unlock keypad to main loop control
	goto	ISR_TMR1_END

ISR_TMR1_rollover
    movlw   LOW(TMR1_INIT)		; reset TMR1
    movwf   TMR1L
    movlw   HIGH(TMR1_INIT)
    movwf   TMR1H
    bcf     PIR1, TMR1IF    	; clear interrupt flag
ISR_TMR1_END

 	; check for GP2 interrup switch
	; -----------------------------
ISR_GP2
	btfsc	INTCON,INTE		; GP2/INT interrupt enable ?
	btfss	INTCON,INTF		; yes: interrupt  GP2/INT running?
	goto 	ISR_GP2_END		; no: next interrupt
	bcf		INTCON,INTE		; disable GP2/INT interrupt
	bcf		INTCON,INTF		; clear interrupt flag
    bcf     UCON, MSGF		; unset flag for abort message sending
ISR_GP2_END
ISRnext
	; other interrupt
	goto	ISRend				; end of interrupt

;restore register
ISRend
	swapf	ISRstatus,w		; STATUS without change Z
	movwf   STATUS				
	swapf   ISRwreg,f		; WREG without change Z
 	swapf   ISRwreg,w  			
	retfie  				; return from interrupt set GIE
	
;---------------------------------------------------------------------- 	
	org		MSG_DATA
;---------------------------------------------------------------------- 	
; Message Table to send
; Beware of bloc page (PCLATH change)
; input: ADCVthPointer = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
SERVICE_MSG_TABLE
	movlw 	HIGH(MSG_DATA) ; PCLATH = HIGH bit address
	movwf 	PCLATH 			;		
	movf	TabPointer,W	; for PCL
	addwf	PCL ,F 			; Jump to character pointed to in W register
SERVICE_MSG
;
; SERVICE MESSAGE
;
IAMBIC_MSG		; iambic key
	DT "I",_EOM
STRAIGHT_MSG	; straight key
	DT "S",_EOM
SPEED_UP_MSG	; signal for speed change up
	DT "U",_EOM
SLOW_DOWN_MSG	; signal for speed change down
	DT "D",_EOM
MOD_A_MSG		; signal for mode A change
	DT "A",_EOM
MOD_B_MSG		; signal for mode B change
	DT "B",_EOM
BUZ_MSG			; signal for sidetone buzzer on/off
	DT _AR,_EOM
ERROR_HM_NOT_FOUND	; signal for error header message not found
	DT _ERR,_EOM
;
	IF ((HIGH ($)) != (HIGH (SERVICE_MSG)))
    	ERROR "SERVICE_MSG_TABLE CROSSES PAGE BOUNDARY!"
	ENDIF;

;---------------------------------------------------------------------- 	
	org		OSCCAL_FACTORY_ADR
;---------------------------------------------------------------------- 	

;---------------------------------------------------------------------- 	
; EEPROM                        
;---------------------------------------------------------------------- 	
	org 	EEPROM_DATA	
EE_NAME			DE	"F8EOZ"
EE_VERSION		DE	"1000070115",0
EE_WPM			DE	SPEED_DEFAULT
EE_UCON			DE	UCON_DEFAULT

;----------------------------------------
	ifdef __ADRES_TRON	
; ADC value report for analyze
; 3x3*2bytes: MSB,LSB
EE_ADRES1to9	DE  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
EE_ADRESH0		DE	0	; MSB
EE_ADRESL0		DE	0	; LSB
	endif
;----------------------------------------

;--------------
; MESSAGE BLOCK
;--------------
; Your call sign here
EE_CALLSIGN	; don't remove this label	
;
	DE "F8EOZ" ; your call sign here CAPS LOCK ONLY
	DE _EOM ; end of call sign don't remove

;----------------------
; Prerecorded message#:
; A..Z CAPS LOCK ONLY
; @ alias for CALLSIGN
EE_MESSAGE ; don't remove this label
;
; message#0:
; CQ CQ CQ DE F8EOZ F8EOZ... 
	DE _HOM,KPAD_SEND0,0x03 ; Head of message directive: message number (button assign), beacon time (0xFF=infinity) 
	DE "CQ CQ CQ DE @ @ K "
	DE _EOM ; end of message#0
;
; message#1:
; CQ DX CQ DX CQ DX DE F8EOZ F8EOZ... 
	DE _HOM,KPAD_SEND1,0x03 ; Head of message directive: message number (button assign), beacon time (0xFF=infinity) 
	DE "CQ DX CQ DX CQ DX DE @ @ K "
	DE _EOM ; end of message#1
;
; message#2:
; CQ CQ F8EOZ F8EOZ TEST ... 
	DE _HOM,KPAD_SEND2,0x03 ; Head of message directive: message number (button assign), beacon time (0xFF=infinity) 
	DE "CQ CQ @ @ TEST "
	DE _EOM ; end of message#2
;----------------------------------------
;----------------------------------------
;----------------------------------------
	end
; EOF: keyer12F675-10.asm