;******************************************************************************
; File Name    	: usartELoader-10.asm
; Version      	: 1.0
; Title:       	: USART EEPROM Loader+BLINK LED demo
; Author		: F8EOZ Bernard DECAESTECKER http://f8eoz.com
; MCU			: Written and Tested on Microchip PIC 16F628 Microcontroller
; Compiler     	: Microchip MPASM Assembler V5.51
; IDE          	: Microchip MPLAB IDE v8.92
; Programmer   	: Home made
; Last Updated 	: 22 September 2015
; *******************************************************************
; Hardware:     
;  Tested with PIC16F628 at 4Mhz  XTAL oscillator.       
;  USART use pin 8 as RX (RB1) 
;  USART use pin 9 as TX (RB2) 
; *******************************************************************
; Description: 
; *******************************************************************
; ELoader is built as a Boot Loader
; ELoader is a small piece of firmware, stored in bottom of
; microcontroller's memory, which can write to the EEPROM of 
; that microcontroller. 
; It uses simple generic asynchronous half-duplex communication protocol:
; RX frame from Host:
; 	"L" : Login (26s Timeout at Reset customisable)
; 	"W"aabbccddee : Write Data ccdd into EEPROM @Address aabb ee=cheksum
; 	"P"aabb : Print Data from EEPROM @Address aabb
; 	"X" : Logout and run user program
; TX frame to Host:
; 	"L"ccdd : Login acknowledgment ccdd = EEPROM start address
; 	"K" : Write acknowledgment
; 	"P"ccdd : Data ccdd to print
; 	"!" : Writing data error
; 	"?" : Writing cheksum error
; 	"$" : Writing Boundary error
; PARAMETERS see below:
;   Settle depending on PIC
;   EEPROM_SIZE : Maximum EEPROM bytes
;   FLASH_PROG_SIZE : FLASH PROGRAM bytes
;
;   Choose your Baud Rate with these parameters below:
;	SPBRG_VALUE : Baud Rate
;	SPBRG_BRGH : High or Low speed Baud Rate
;
;   Choose your Login Timeout with these parameters below:
;   ELOADER_TIMER1_OVF  Overflow TIMER1 counter for login timeout
;   ELOADER_TIMER1	Base TIMER1 value
;   ELOADER_T1CKPS  Prescale value
; *******************************************************************
	TITLE "usartELoader-10.asm"
	errorlevel -302, -306		; no message 302 & 306

;====================================================================== 	
;====================================================================== 	
; USER PROGRAMM DEFINES
;====================================================================== 	
;====================================================================== 	
	list p=16f628a	
	#include <p16f628a.inc>

  	__CONFIG (_WDT_OFF & _CP_OFF & _LVP_OFF & _PWRTE_ON & _HS_OSC)

#define ISR_VECTOR			0x0004	; Address of Interrupt Vector
#define COMMON_RAM_DATA		0x0020	; 16 General Purpose registers (SRAM)

#DEFINE LED	PORTA,2			; LED 

;====================================================================== 	
;====================================================================== 	
; END USER PROGRAMM DEFINES
;====================================================================== 	
;====================================================================== 	

;---------------------------------------------------------------------- 	
; EEPROM LOADER DEFINES 
;---------------------------------------------------------------------- 	

;------------------------- 	
;	Start Programm Address
#define RESET_VECTOR 0x0000	; Address of RESET Vector

;------------------------- 	
; Memory PIC features
#DEFINE FLASH_PROG_SIZE 0x7FF ; 2K FLASH PROGRAM
#DEFINE EEPROM_SIZE H'7F' ; EEPROM = 128 bytes

;------------------------- 	
; Test and Debug
#DEFINE ELOADER_TEST 0

;-----------------------------------
; SPRG: BAUD RATE GENERATOR REGISTER
; FOSC  = 4MHz XTAL
; SPBRG = (FOSC/(Desired Baud Rate*16))  1	
; SPBRG = (4000000/(9600*16))  1 = 25,04 = 25
; Calculated Baud Rate = 4000000/(16 * (25+1)) = 9615,38 bauds
; Error = (9615,38  9600) / 9600 = 0,16% <	4%
SPBRG_VALUE = D'25'
#define SPBRG_BRGH 1

;--------------------------------------
; LOGIN TIMEOUT PARAMETERS
; Instruction cycle = FOSC/4 = 1s @4MHz
; 65536 * 2 * 200 = 26214400us = 26s
; 65536 = TIMER1 0 to 65535
; 2 = prescale value
; 200 = roll over TIMER1
#define ELOADER_TIMER1_OVF  D'200'
ELOADER_TIMER1	EQU	D'65536'
#define ELOADER_T1CKPS H'10' ; Prescale=1:2
;#define ELOADER_T1CKPS H'00' ; Prescale=1:1
;#define ELOADER_T1CKPS H'20' ; Prescale=1:4
;#define ELOADER_T1CKPS H'30' ; Prescale=1:8

	IF ELOADER_TIMER1_OVF == 0
		ERROR "ELOADER_TIMER1_OVF Overflow time parameter must be > 0"
	ENDIF

#define ELoaderSize	0xA2 ; EEPROM Loader size
;#define ELoader	ELoaderExit+5		     ; adresse de base du bootloader
;#define ELoaderAdrEnd	FLASH_PROG_SIZE 
#define ELoaderRoot	(FLASH_PROG_SIZE)-ELoaderSize+1 ; EEPROM Loader Root address
;#define NumRetries	1			     ; number of writing retries
; Communication Query Code
#define ELOADER_WRITE	    		"W" ; Query Write into RAM			 
#define ELOADER_WRITE_EEPROM		"E" ; Query Write into EEPROM			 
#define ELOADER_PRINT_EEPROM		"P" ; Query Print EEPROM			 
#define ELOADER_WRITE_ACK	   		"K" ; Write OK
#define ELOADER_WRITE_ERR	    	"!" ; Write KO data error
#define ELOADER_WRITE_BOUNDARY_ERR	"$" ; Write KO boundary error
#define ELOADER_CHEKSUM_ERR    		"?" ; Write KO Cheksum rror
#define ELOADER_LOGIN				"L" ; Query Login to device
#define ELOADER_LOGIN_ACK  			"L" ; Connect Login to device Acknowledge
#define ELOADER_LOGOUT				"X" ; Logout and Run Program
#define NEW_LINE 					0x0A ; End of a TX line

;-----------------------------------------------------------------------------
; EEPROM LOADER FILES REGISTER
; This data can be redefined by program
#define ELOADER_CBLOCK 0x70 ; Common RAM

; Query BUFFER 6 Bytes
ELoaderBuffer   EQU ELOADER_CBLOCK
ELoaderFiller0  EQU ELOADER_CBLOCK
ELoaderFiller1	EQU ELOADER_CBLOCK+1
ELoaderFiller2	EQU ELOADER_CBLOCK+2
ELoaderFiller3	EQU ELOADER_CBLOCK+3
ELoaderFiller4	EQU ELOADER_CBLOCK+4
ELoaderFiller5	EQU ELOADER_CBLOCK+5
; Query LOGIN Refefines BUFFER: "L"
; Query LOGOUT Refefines BUFFER: "X"
; Query WRITE EEPROM Redefines BUFFER: "W"aabbccddee
; Query PRINT EEPROM Redefines BUFFER: "P"aabb
ELoaderQcode    EQU ELOADER_CBLOCK	 ;Query Code
ELoaderAddrH    EQU ELOADER_CBLOCK+1 ;aa
ELoaderAddrL    EQU ELOADER_CBLOCK+2 ;bb
ELoaderDataH    EQU ELOADER_CBLOCK+3 ;cc
ELoaderDataL    EQU ELOADER_CBLOCK+4 ;dd
ELoaderChkSum   EQU ELOADER_CBLOCK+5 ;ee

ELoaderOVFtime	EQU	ELOADER_CBLOCK+9

;---------------------------------------------------------------------- 	
; END EEPROM LOADER DEFINES 
;---------------------------------------------------------------------- 	

;====================================================================== 	
;====================================================================== 	
; USER RAM FILES
;====================================================================== 	
;====================================================================== 	
	cblock COMMON_RAM_DATA
	;------------------ 	
	; Delay count Files
	cmpt1 : 1			
	cmpt2 : 1			
	cmpt3 : 1	
	;---------------- 	
	; Interrupt Files
	ISRwreg:1			; save WREG
	ISRstatus:1			; save STATUS
	endc				; End of General Purpose registers                     
;====================================================================== 	
;====================================================================== 	
; END USER RAM FILES
;====================================================================== 	
;====================================================================== 	

	ORG		RESET_VECTOR  ; RESET Vector
	nop				      
	pagesel ELoaderStart
	goto	ELoaderStart
	org		ISR_VECTOR    ; interrupt vector
	goto	ISR

;====================================================================== 	
;====================================================================== 	
; USER PROGRAM ROOT 
;====================================================================== 	
;====================================================================== 	
UserProgStart
;------------
Main
	clrf	PORTA			
	clrf	PORTB			
	bsf		STATUS,RP0		; Bank1
	clrf	EEADR			
	; set up option register
	movlw	B'00001000'
    ;         0-------  NOT RBPU: 0 = PORTB pull-ups are enabled by individual port latch values
    ;         -0------  INTEDG: 0 = Interrupt on rising edge of RB0/INT pin
    ;         --0-----  T0CS: 1 = Transition on GP2/T0CKI pin0=output, GP4, Buzzer control
    ;         ---0----  T0SE: 1 = Increment on high-to-low transition on GP2/T0CKI pinnot used, GP3, MCLR and reset
    ;         ----1---  PSA: 1 = Prescaler is assigned to the WDT
    ;         -----000  PS2:PS0: 111= 1:128 prescaler
	movwf	OPTION_REG
	bcf		LED				; TRISA output
	bcf		STATUS, RP0		; Bank0

MainLoop
	bsf	LED			   	; LED on			
	call 	delay	
	bcf	LED			   	; LED off			
	call 	delay	

	goto	MainLoop	; For Ever Loop------>

;------------------------- 	
; USER PROGRAM SUBROUTINES 
;------------------------- 	
delay
	movlw	2			
	movwf	cmpt3		
delay3
	clrf	cmpt2		
delay2
	clrf	cmpt1		
delay1
	nop					
	decfsz 	cmpt1, f	
	goto 	delay1		
	decfsz 	cmpt2, f	
	goto 	delay2		
	decfsz 	cmpt3, f	
	goto 	delay3		
	return

;------------------ 	
; INTERRUPT MANAGER
;------------------ 	
ISR
	;save register
	movwf	ISRwreg  		; WREG
	swapf	STATUS,w		; STATUS without change Z
	movwf	ISRstatus		; 
;restore register
ISRend
	swapf	ISRstatus,w		; STATUS without change Z
	movwf   STATUS				
	swapf   ISRwreg,f		; WREG without change Z
 	swapf   ISRwreg,w  			
	retfie  				; return from interrupt set GIE
;====================================================================== 	
;====================================================================== 	
; USER PROGRAM END
;====================================================================== 	
;====================================================================== 	


;---------------------------------------------------------------------- 	
; EEPROM LOADER ROOT 
;---------------------------------------------------------------------- 	
	ORG	ELoaderRoot

ELoaderExit			       
	clrf	PCLATH	       ; Don't remove set PCLATH on PAGE 0

	; Set PCLATH on user program start page and run
	;pagesel ELoaderExit	 
	;goto 	ELoaderExit
	pagesel UserProgStart
	goto 	UserProgStart ;>>>>>>>>>>>>>>>EXIT

;--------------
	ORG	ELoaderRoot+5

ELoaderStart
	btfss	STATUS,NOT_TO	; WatchDog-TimeOut ?
	goto	ELoaderExit

	;-------------
	; Settle USART
	;-------------
	; set up RCSTA: RECEIVE STATUS AND CONTROL REGISTER	
	BANKSEL	RCSTA
	movlw	B'10010000'
    ;         1-------  SPEN 1=Serial port enabled
    ;         -0------  RX9 0=Selects 8-bit reception
    ;         --0-----  SREN 0=don't care in asynchronous mode
    ;         ---1----  CREN 1=in Asynchronous mode, Enables continuous receive
    ;         ----0---  ADEN 0=in Asynchronous mode 9-bit (RX9 = 1):	
	;						Disables address detection, all bytes are received, 
	;						and ninth bit can be used as PARITY bit	
    ;         -----0--  FERR 0=No framing error
    ;         ------0-  OERR 0=No overrun error
    ;         -------0  RX9D 0=9th bit of received data. Can be PARITY bit	
	movwf	RCSTA

	; set up TXSTA: TRANSMIT STATUS AND CONTROL REGISTER	
	BANKSEL	TXSTA
	movlw	B'00100100'
    ;         0-------  CSRC 0=don't care in asynchronous mode
    ;         -0------  TX9 0=Selects 8-bit transmission
    ;         --1-----  TXEN 0=Transmit enabled
    ;         ---0----  SYNC 0=Asynchronous mode
    ;         ----0---  unused
    ;         -----1--  BRGH 1=In asynchronous mode High speed
    ;         ------0-  TRMT 0=TSR full
    ;         -------0  TX9D 0=9th bit of transmit data. Can be PARITY bit	
	movwf	TXSTA

	IF SPBRG_BRGH != 1     ; USART SYNC=0; SPEN=1; CREN=1; SREN=0;
		bcf TXSTA,BRGH
	ENDIF

	; Set up SPRG: Baud Rate Generator Register
	movlw	SPBRG_VALUE 
	movwf	SPBRG
	
	;----------------
	; LOGIN COUNTDOWN
	;----------------
	bcf		STATUS,RP0
	movlw	ELOADER_TIMER1_OVF
	movwf	ELoaderOVFtime
	; Settle TIMER1 CONTROL REGISTER	
	movlw	B'00000000' | ELOADER_T1CKPS ; Prescaler value depending on FOSC
    ;         00------  Unimplemented: Read as '0'
    ;         --00----  T1CKPS1:T1CKPS0: Timer1 Input Clock Prescale Select bits
	;						11 = 1:8 Prescale value	
	;						10 = 1:4 Prescale value	
	;						01 = 1:2 Prescale value	
	;						00 = 1:1 Prescale value	
    ;         ----0---  T1OSCEN: Timer1 Oscillator Enable Control bit
	;						0 = Oscillator is shut off	
    ;         -----0--  NOT T1SYNC: Timer1 External Clock Input Synchronization Control bit
	;					This bit is ignored. Timer1 uses the internal clock when TMR1CS = 0. 	
    ;         ------0-  TMR1CS: Timer1 Clock Source Select bit
	;						0 = Internal clock (FOSC/4) 	
    ;         -------0  TMR1ON: Timer1 On bit
	;						1 = Enables Timer1 
	;						0 = Stops Timer1 	
	movwf	T1CON

	; TMR1 Overflow Interrupt Flag not overflowed 	
	bcf    	PIR1,TMR1IF
	movlw  	high ELOADER_TIMER1
	movwf  	TMR1H		       
	movlw  	low ELOADER_TIMER1
	movwf  	TMR1L		       
	bsf    	T1CON,TMR1ON	; TIMER1 On

	call	ELoaderWaitForLogin
	xorlw	ELOADER_LOGIN
	btfss	STATUS,Z
	goto	ELoaderLogout
	movlw	ELOADER_LOGIN	; Restore Query code

;------------------------------------------------------------
ELoaderLoop 				   
;----------

	; switch case ELoaderQcode
	;
	; case: WRITE INTO FLASH RAM
	movwf  	ELoaderQcode
	xorlw  	ELOADER_WRITE		; Unimplemented
	btfsc  	STATUS,Z
	goto   	ELoaderWrite

	; case: WRITE INTO EEPROM
	xorlw  	ELOADER_WRITE^ELOADER_WRITE_EEPROM
	btfsc  	STATUS,Z
	goto   	ELoaderWrite		   

	; case: PRINT EEPROM
	xorlw  	ELOADER_WRITE_EEPROM^ELOADER_PRINT_EEPROM
	btfsc  	STATUS,Z
	goto   	ELoaderPrintEEPROM	

	; case: LOGIN
	xorlw  	ELOADER_PRINT_EEPROM^ELOADER_LOGIN
	btfsc  	STATUS,Z
	goto   	ELoaderLogin		 

	; case: LOGOUT
	xorlw  	ELOADER_LOGIN^ELOADER_LOGOUT
	btfsc  	STATUS,Z 		    
	goto	ELoaderLogout

ELoaderLoopRead
	call   	ELoaderReadByte	; Wait for a Byte from USART
	goto   	ELoaderLoop

;------------
; LOGOUT
ELoaderLogout
	; Stop TIMER1
	clrf  	T1CON			       
	; Reset USART Registers
	clrf  	RCSTA
	bsf   	STATUS,RP0
	clrf  	TXSTA			     
	bcf   	STATUS,RP0
	clrf  	PIR1
	;---------------------
	; Program User Execute
	goto  	ELoaderExit 
	;---------------------

;------------------------------
; LOGIN acknowledgment
; Send start address of ELoader
; TX : Phhll
ELoaderLogin
	movlw	ELOADER_LOGIN ; LOGIN acknowledgment
	call	ELoaderWriteByte
	movlw	high (ELoaderRoot & 0x7FF)
	call	ELoaderWriteByte
	movlw	low (ELoaderRoot & 0x7FF)

	goto	ELoaderWriteLine

;----------------
; Not implemented
ELoaderWriteFlash
	goto	ELoaderLoopRead

;-------------------------
; PrintEEPROM
; Send 2 bytes from EEPROM
; TX Format: Phhll
ELoaderPrintEEPROM
	; Data Address HIGH
	call	ELoaderReadByte
	movwf	ELoaderAddrH
	; Data Address LOW
	call	ELoaderReadByte
	movwf	ELoaderAddrL
	xorlw  	EEPROM_SIZE+1 ; Is End of EEPROM ?
	btfsc  	STATUS,Z
	goto	ELoaderPrintEEPROM_send ; Yes

	bsf		STATUS,RP0		; Bank1
	movfw	ELoaderAddrL
	movwf	EEADR 
	bsf		EECON1,RD		; read
	movf	EEDATA,W		; data ouput Z=1 if zero

	bcf		STATUS,RP0		; Bank0
	movwf	ELoaderDataL
	clrf	ELoaderDataH	; nothing
	; TX : Paabb
ELoaderPrintEEPROM_send
	movlw	ELOADER_PRINT_EEPROM
	call	ELoaderWriteByte ;<<<<<<<<<<<<<<<<<<
	movfw	ELoaderDataH
	call	ELoaderWriteByte ;<<<<<<<<<<<<<<<<<
	movfw	ELoaderDataL

	goto	ELoaderWriteLine	

;------------------------
; Common Write processing 
ELoaderWrite
	; Data Address HIGH
	call	ELoaderReadByte
	movwf	ELoaderAddrH
	movwf	ELoaderChkSum	;init checksum 
	; Data Address LOW
	call	ELoaderReadByte
	movwf	ELoaderAddrL
	addwf	ELoaderChkSum,f	;+ Checksum

	; Data byte HIGH
	call	ELoaderReadByte
	movwf	ELoaderDataH	; hold data into buffer
	addwf	ELoaderChkSum,f ;+ Checksum

	; Data byte LOW
	call	ELoaderReadByte
	movwf	ELoaderDataL	; hold data into buffer
	addwf	ELoaderChkSum,f ;+ Checksum

	; Checksum
	call	ELoaderReadByte 		
	xorwf	ELoaderChkSum,w	; chksum = w ?
	movlw	ELOADER_CHEKSUM_ERR
	btfss	STATUS,Z
	goto	ELoaderWriteLine; checksum error

	; checksum_ok
	; Split Write Processing: Flash RAM or EEPROM
	movfw	ELoaderQcode
	xorlw	ELOADER_WRITE_EEPROM
	btfsc	STATUS,Z
	goto	ELoaderWriteEEPROM	; write byte into eeprom

	goto	ELoaderWriteFlash

;-----------------
ELoaderWriteEEPROM
	; Write byte into EEPROM @Address
	movfw	ELoaderDataL
	bsf		STATUS,RP0		; Bank1
	movwf	EEDATA			
	movf	ELoaderAddrL,w			
	movwf	EEADR				
	bsf		EECON1,WREN		; Allows write cycles	
	; Write Sequence
	movlw	0x55			
	movwf	EECON2			
	movlw	0xAA			
	movwf	EECON2			
	bsf		EECON1,WR		; Initiates a write cycle	
	
	; Wait for write end
	clrwdt					
	btfsc	EECON1,WR		; is done?
	goto	$-2				; no: loop wait
	bcf		EECON1,WREN		; yes: Inhibits write to the data EEPROM	

	; Write verify
	movf	EEDATA,w
	bsf		EECON1,RD		; read the data written
	xorwf	EEDATA,w
	bcf		STATUS,RP0		; Bank0
	movlw	ELOADER_WRITE_ERR			
	btfss	STATUS,Z		; Data read = data written?
	goto	ELoaderWriteLine; Write error

	; Write OK
	movlw	ELOADER_WRITE_ACK			

	goto	ELoaderWriteLine

;---------------------------------------------------------------------- 	
; Write one byte + new line to USART
; Input: WREG = Byte to send
ELoaderWriteLine
	;goto    ELoaderLoop ;<<<<<<<<<<<<<<<<<<<<<<<<<<
	; Send byte + end of line
	call	ELoaderWriteByte
	movlw	NEW_LINE
	call	ELoaderWriteByte

	goto	ELoaderLoopRead ; Next byte>>>>>>>>>>

;---------------------------------------------------------------------- 	
; Write one Byte to USART 
; Input: WREG = Byte to send
ELoaderWriteByte
	clrwdt
	btfss	PIR1,TXIF		       ; while(!TXIF)
	goto	$-2
	movwf	TXREG			       ; TXREG = octet
	return

;---------------------------------------------------------------------- 	
; Read one Byte from USART 
; Output: WREG = Received Byte
ELoaderReadByte
	clrwdt
	btfss	PIR1,RCIF		       
	goto	$-2
	movf	RCREG,w	; Hold byte from RCREG into WREG
	return

;----------------------------------------------------- 	
; Wait for login from Host 
; Output: WREG = 'L' login or 'X' Execute user program
ELoaderWaitForLogin
	clrwdt
	btfss	PIR1,RCIF		       
	goto	$+3		; Nothing received: continue
	movf	RCREG,w ; Hold received byte RCREG into WREG
	return			; Exit OK >>>>>>>>>>>>>>

	btfss  	PIR1,TMR1IF	       		; TIMER1 overflow?
	goto   	ELoaderWaitForLogin 	; No: loop---------->
	bcf    	T1CON,TMR1ON	      	; Yes: TIMER1 Off
	decfsz 	ELoaderOVFtime,f		
	goto   	ELoaderWaitForLoginOVF	; Reset TIMER1------>
	retlw 	'X'    	; Time out WREG=EXIT >>>>>>>>>>>>>>>

	; Set TIMERH
ELoaderWaitForLoginOVF
	bcf    	PIR1,TMR1IF
	movlw  	high ELOADER_TIMER1
	movwf  	TMR1H		       
	bsf    	T1CON,TMR1ON			; TIMER1 On
	goto	ELoaderWaitForLogin
     
;--------------------------
	ORG 	FLASH_PROG_SIZE
;--------------------------
	END