;******************************************************************************
; File Name    	: chargePIC16F88.asm
; Version      	: 1.0
; Title:       	: USART Bootloader
; Author		: F8EOZ Bernard DECAESTECKER http://f8eoz.com
; MCU			: Written and Tested on Microchip PIC 16F88 Microcontroller
; Compiler     	: Microchip MPASM Assembler V5.51
; IDE          	: Microchip MPLAB IDE v8.92
; Programmer   	: Home made
; Last Updated 	: 11 October 2015
; *******************************************************************
; Hardware:     
;  Tested with PIC16F88 at 4Mhz  internal oscillator.       
;  USART use pin 8 as RX (RB2) 
;  USART use pin 11 as TX (RB5) 
; *******************************************************************
; Description: 
; *******************************************************************
; BLoader is a small piece of firmware, stored in bottom of
; microcontroller's memory, which can write to the EEPROM of 
; that microcontroller. 
; It uses simple generic asynchronous half-duplex communication protocol:
; LOGIN		     0x01
; WRITE_PROG	 0x02
; WRITE_EEPROM   0x04
; READ_PROG	     0x08
; READ_EEPROM    0x10
; ERASE_PROG	 0x20
; LOGOUT		 0x80
; PARAMETERS see below:
;   Settle depending on PIC
;   EEPROM_SIZE : Maximum EEPROM bytes
;   PROG_MEM_SIZE : FLASH PROGRAM bytes
;
;   Choose your Baud Rate with these parameters below:
;	SPBRG_VALUE : Baud Rate
;	SPBRG_BRGH : High or Low speed Baud Rate
;
;   Choose your Login Timeout with these parameters below:
;   BLOADER_TIMER1_OVF  Overflow TIMER1 counter for login timeout
;   BLOADER_TIMER1	Base TIMER1 value
;   BLOADER_T1CKPS  Prescale value
; *******************************************************************
; Erase/Write note:
; 16F88 requires explicit 32 words block erase and then 4 words block write
;
; Bootloader Mapping:
; x000 - 0x01F: Offset: reset segment
; xF00 - 0xFFF: Bootloader
; x020 - 0xEFF: User Application
; Note: User Aplication space contains a test blink led program on RA2
; *******************************************************************
; User application Template

;---------------------------------------------------------------------- 	
; BOOTLOADER RESET SEGMENT RESERVED 32 WORDS
;---------------------------------------------------------------------- 	
;BLOADER_OFFSET	EQU 0x20
;
;	ORG 	RESET_VECTOR
;	goto	RESET_VECTOR+BLOADER_OFFSET+4

;	; interrupt vector
;	ORG		ISR_VECTOR    
;	goto	RESET_VECTOR+BLOADER_OFFSET

;---------------------------------------------------------------------- 	
; END BOOTLOADER RESET SEGMENT RESERVED 32 WORDS
;---------------------------------------------------------------------- 	

;---------------------------------------------------------------------- 	
; USER APPLICATION SEGMENT
;---------------------------------------------------------------------- 	
; Re-mapped Interrupt Vector
;	ORG		RESET_VECTOR+BLOADER_OFFSET
;UserISR
;	;retfie
;	goto	UserInterrupt

; Re-mapped Reset Vector
;	ORG		RESET_VECTOR+BLOADER_OFFSET+4
;UserStart
;	;goto	Main

;---------------------------------------------------------------------- 	
; USER PROGRAM STARTS HERE
;---------------------------------------------------------------------- 	

;Main
;...

; *******************************************************************

;====================================================================== 	
;====================================================================== 	
;====================================================================== 	
;====================================================================== 	

;------------------------- 	
; PIC DEVICE & FUSE
	errorlevel -302	; Turn off banking message known tested (good) code
	errorlevel +302	; Enable banking message untested code
	errorlevel -306	; Turn off Crossing page boundary -- ensure page bits are set
	;errorlevel +306	; Enable Crossing page boundary -- ensure page bits are set
	radix DEC

	TITLE "chargePIC16F88.asm"
	list p=16f88	
	#include <p16f88.inc>

	__CONFIG    _CONFIG1, _INTRC_IO  & _CP_OFF & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO
	__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

;------------------------- 	
; Memory PIC features
PROG_MEM_SIZE = 0x1000 ; 4K FLASH PROGRAM
EEPROM_SIZE = H'100' ; EEPROM = 256 bytes

;------------------------- 	
; Programm Vector  Address
RESET_VECTOR	EQU	0x0000	; Address of RESET Vector
ISR_VECTOR		EQU	0x0004	; Address of Interrupt Vector
BANK0_RAM_DATA	EQU	0x0020	; 16 General Purpose registers (SRAM)
COMMON_RAM_DATA	EQU	0x0070	; 16 General Purpose registers (SRAM)
EEPROM_DATA		EQU	0x2100	; EEPROM address 

;------------------------- 	
; Memory BLoader features
BLOADER_SIZE = H'100' ; Bootloader size = one boundary page
BLOADER_RESET = RESET_VECTOR
BLOADER_OFFSET = H'20' ; 32 WORDS reserved space for relocated user program start address
BLOADER_ROOT = PROG_MEM_SIZE-BLOADER_SIZE ; Bootloader Root address
BLOADER_VERSION = H'0100' ; High = Version Low = revision
USER_RESET = BLOADER_RESET + BLOADER_SIZE ; Bootloader ORG Address
USER_INTERRUPT = BLOADER_RESET + BLOADER_SIZE+ISR_VECTOR ; Bootloader ORG Address
;------------------------- 	
; Test and Debug
BLOADER_TEST = 0

;-----------------------------------
; SPRG: BAUD RATE GENERATOR REGISTER
; FOSC  = 4MHz XTAL
; SPBRG = (FOSC/(Desired Baud Rate*16))  1	
; SPBRG = (4000000/(9600*16))  1 = 25,04 = 25
; Calculated Baud Rate = 4000000/(16 * (25+1)) = 9615,38 bauds
; Error = (9615,38  9600) / 9600 = 0,16% <	4%
SPBRG_VALUE = D'25'
SPBRG_BRGH = 1

;--------------------------------------
; LOGIN TIMEOUT PARAMETERS
; Instruction cycle = FOSC/4 = 1s @4MHz
; 65536 * 2 * 200 = 26214400us = 26s
; 65536 = TIMER1 0 to 65535
; 2 = prescale value
; 200 = roll over TIMER1
BLOADER_TIMER1_OVF  EQU D'200'
BLOADER_TIMER1		EQU	D'65536'
BLOADER_T1CKPS 		EQU H'10' ; Prescale=1:2
;#define BLOADER_T1CKPS H'00' ; Prescale=1:1
;#define BLOADER_T1CKPS H'20' ; Prescale=1:4
;#define BLOADER_T1CKPS H'30' ; Prescale=1:8

	IF BLOADER_TIMER1_OVF == 0
		ERROR "BLOADER_TIMER1_OVF Overflow time parameter must be > 0"
	ENDIF

; Communication Error Code
#define BLOADER_CHECKSUM_ERR	"?" ; Write KO Checksum rror

; End of Tx Frame
#define NEW_LINE 					0x0A ; End of a TX line

;------------------------------
; Communication Query Code Bits
LOGIN		 EQU H'0000' 	; 0x01
WRITE_PROG	 EQU H'0001'	; 0x02
WRITE_EEPROM EQU H'0002'	; 0x04
READ_PROG	 EQU H'0003'	; 0x08
READ_EEPROM	 EQU H'0004'	; 0x10
ERASE_PROG	 EQU H'0005'	; 0x20
LOGOUT		 EQU H'0007'	; 0x80
;-----------------------------------------------------------------------------
; BLOADER FILES REGISTER
; This data can be redefined by program
BLOADER_CBLOCK	= COMMON_RAM_DATA ; Common RAM

; Query BUFFER 7 Bytes
BLoaderBuffer   EQU BLOADER_CBLOCK
BLoaderFiller0  EQU BLOADER_CBLOCK
BLoaderFiller1	EQU BLOADER_CBLOCK+1
BLoaderFiller2	EQU BLOADER_CBLOCK+2
BLoaderFiller3	EQU BLOADER_CBLOCK+3
BLoaderFiller4	EQU BLOADER_CBLOCK+4
BLoaderFiller5	EQU BLOADER_CBLOCK+5
BLoaderFiller6	EQU BLOADER_CBLOCK+6

BLoaderQcode    EQU BLOADER_CBLOCK	 ;Query Code
BLoaderAddrH    EQU BLOADER_CBLOCK+1 ;aa
BLoaderAddrL    EQU BLOADER_CBLOCK+2 ;bb
BLoaderDataH    EQU BLOADER_CBLOCK+3 ;cc
BLoaderDataL    EQU BLOADER_CBLOCK+4 ;dd
BLoaderChkSum   EQU BLOADER_CBLOCK+5 ;ee

BLoaderWord		EQU BLOADER_CBLOCK+6 ;Write Word Count

BLoaderOVFtime	EQU	BLOADER_CBLOCK+9

;====================================================================== 	
;====================================================================== 	
; USER RAM FILES
;====================================================================== 	
;====================================================================== 	

	cblock BANK0_RAM_DATA
	;-------------------- 	
	; Delay count Files
	cmpt1 : 1			
	cmpt2 : 1			
	cmpt3 : 1	
	;---------------- 	
	; Interrupt Files
	ISRwreg:1			; save WREG
	ISRstatus:1			; save STATUS
	endc				; End of General Purpose registers                     

	cblock COMMON_RAM_DATA
	;--------------------- 	
	endc				; End of General Purpose registers                     

;====================================================================== 	
;====================================================================== 	
; END RAM FILES
;====================================================================== 	
;====================================================================== 	


;---------------------------------------------------------------------- 	
; BOOTLOADER RESET SEGMENT RESERVED 32 WORDS
;---------------------------------------------------------------------- 	
	ORG 	RESET_VECTOR

	PAGESEL BLOADER_ROOT
	GOTO    BLOADER_ROOT
	nop
	nop
	
	; interrupt vector
	ORG		ISR_VECTOR    
	goto	UserISR
	
	ORG		BLOADER_RESET+8
	nop		; reserved for extention
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
BLoaderExit
	;------------------------------------
	;goto	RESET_VECTOR+BLOADER_OFFSET
;---------------------------------------------------------------------- 	
; END BOOTLOADER RESET SEGMENT RESERVED 32 WORDS
;---------------------------------------------------------------------- 	

;---------------------------------------------------------------------- 	
; USER APPLICATION SEGMENT
;---------------------------------------------------------------------- 	

; Re-mapped Reset Vector
	ORG		RESET_VECTOR+BLOADER_OFFSET
UserReset
	goto	UserStart

; Re-mapped Interrupt Vector
	ORG		RESET_VECTOR+BLOADER_OFFSET+4
UserISR
	;retfie
	goto	UserInterrupt

;---------------------------------------------------------------------- 	
; USER PROGRAM STARTS HERE
;---------------------------------------------------------------------- 	
; LED BLINK TEST PROGRAM

UserStart
Main
#DEFINE LED	PORTA,2			; LED 

	banksel	PORTA
	clrf	PORTA			
	clrf	PORTB			
	; set up option register
	bsf		STATUS,RP0		; Bank1
	movlw	B'00001000'
    ;         0-------  NOT RBPU: 0 = PORTB pull-ups are enabled by individual port latch values
    ;         -0------  INTEDG: 0 = Interrupt on rising edge of RB0/INT pin
    ;         --0-----  T0CS: 1 = Transition on GP2/T0CKI pin0=output, GP4, Buzzer control
    ;         ---0----  T0SE: 1 = Increment on high-to-low transition on GP2/T0CKI pinnot used, GP3, MCLR and reset
    ;         ----1---  PSA: 1 = Prescaler is assigned to the WDT
    ;         -----000  PS2:PS0: 111= 1:128 prescaler
	movwf	OPTION_REG
	bcf		LED				; TRISA output
	bcf		STATUS, RP0		; Bank0

MainLoop
	bsf	LED			   	; LED on			
	call 	delay	
	bcf	LED			   	; LED off			
	call 	delay	

	goto	MainLoop	; For Ever Loop------>

;------------------------- 	
; USER PROGRAM SUBROUTINES 
;------------------------- 	
delay
	movlw	2			
	movwf	cmpt3		
delay3
	clrf	cmpt2		
delay2
	clrf	cmpt1		
delay1
	nop					
	decfsz 	cmpt1, f	
	goto 	delay1		
	decfsz 	cmpt2, f	
	goto 	delay2		
	decfsz 	cmpt3, f	
	goto 	delay3		
	return

;------------------ 	
; INTERRUPT MANAGER
;------------------ 	
UserInterrupt
	retfie  				; return from interrupt set GIE
;---------------------------------------------------------------------- 	
; EEPROM                        
;---------------------------------------------------------------------- 	
	org 	EEPROM_DATA	

;----------------------------------------
;              Month Day   Year  Vnum
EE_VERSION	DE D'12',D'17',D'15',1,0,2
EE_SETTING	DE "chargePIC: TEST ONE BLINK LED ON RA2 PORT"
EE_BLINK_F  DE 0xFE
;---------------------------------------------------------------------- 	
; END USER APPLICATION SEGMENT
;---------------------------------------------------------------------- 	


;---------------------------------------------------------------------- 	
; BOOTLOADER SEGMENT RESREVED 256 WORDS
;---------------------------------------------------------------------- 	
	ORG 	BLOADER_ROOT

BLoaderStart
	; set up osc internal 8 Mhz	
	BANKSEL	OSCCON ;BANK1_
	movlw	b'01100100'		
    ;         0-------  Unimplemented: Read as 0
    ;         -110----  IRCF<2:0>: Internal RC Oscillator Frequency Select bits 
	;						000 = 31.25 kHz 
	;						001 = 125 kHz
	;						010 = 250 kHz 
	;						011 = 500 kHz 	
	;						100 = 1 MHz 
	;						101 = 2 MHz 	
	;						110 = 4 MHz 	
	;						111 = 8 MHz  	
    ;         ----0---  OSTS: Oscillator Start-up Time-out Status bit 	
	;						1 = Device is running from the primary system clock
	;						0 = Device is running from T1OSC or INTRC as a secondary system clock 
	;					Note 1: Bit resets to 0 with Two-Speed Start-up mode and LP, XT or HS selected as the	oscillator mode.  
    ;         -----1--  IOFS: INTOSC Frequency Stable bit  	
	;						1 = Frequency is stable 
	;						0 = Frequency is not stable
    ;         ------00  SCS<1:0>: Oscillator Mode Select bits 
	;						00 = Oscillator mode defined by FOSC<2:0> 
	;						01 = T1OSC is used for system clock 
	;						10 = Internal RC is used for system clock
	;						11 = Reserved	
	movwf	OSCCON
	; Wait fot osc stable (only for internal osc)
	btfss	OSCCON,IOFS
	goto $-1

	;-------------
	; Settle USART
	;-------------
	; set up RCSTA: RECEIVE STATUS AND CONTROL REGISTER	
	BANKSEL	RCSTA
	movlw	B'10010000'
    ;         1-------  SPEN 1=Serial port enabled
    ;         -0------  RX9 0=Selects 8-bit reception
    ;         --0-----  SREN 0=don't care in asynchronous mode
    ;         ---1----  CREN 1=in Asynchronous mode, Enables continuous receive
    ;         ----0---  ADEN 0=in Asynchronous mode 9-bit (RX9 = 1):	
	;						Disables address detection, all bytes are received, 
	;						and ninth bit can be used as PARITY bit	
    ;         -----0--  FERR 0=No framing error
    ;         ------0-  OERR 0=No overrun error
    ;         -------0  RX9D 0=9th bit of received data. Can be PARITY bit	
	movwf	RCSTA

	; set up TXSTA: TRANSMIT STATUS AND CONTROL REGISTER	
	BANKSEL	TXSTA
	movlw	B'00100100'
    ;         0-------  CSRC 0=don't care in asynchronous mode
    ;         -0------  TX9 0=Selects 8-bit transmission
    ;         --1-----  TXEN 0=Transmit enabled
    ;         ---0----  SYNC 0=Asynchronous mode
    ;         ----0---  unused
    ;         -----1--  BRGH 1=In asynchronous mode High speed
    ;         ------0-  TRMT 0=TSR full
    ;         -------0  TX9D 0=9th bit of transmit data. Can be PARITY bit	
	movwf	TXSTA

	IF SPBRG_BRGH != 1     ; USART SYNC=0; SPEN=1; CREN=1; SREN=0;
		bcf TXSTA,BRGH
	ENDIF

	; Set up SPRG: Baud Rate Generator Register
	movlw	SPBRG_VALUE 
	movwf	SPBRG
	
	; Init block word counter
	clrf	BLoaderWord

	;----------------
	; LOGIN COUNTDOWN
	;----------------
	bcf		STATUS,RP0
	movlw	BLOADER_TIMER1_OVF
	movwf	BLoaderOVFtime
	; Settle TIMER1 CONTROL REGISTER	
	movlw	B'00000000' | BLOADER_T1CKPS ; Prescaler value depending on FOSC
    ;         00------  Unimplemented: Read as '0'
    ;         --00----  T1CKPS1:T1CKPS0: Timer1 Input Clock Prescale Select bits
	;						11 = 1:8 Prescale value	
	;						10 = 1:4 Prescale value	
	;						01 = 1:2 Prescale value	
	;						00 = 1:1 Prescale value	
    ;         ----0---  T1OSCEN: Timer1 Oscillator Enable Control bit
	;						0 = Oscillator is shut off	
    ;         -----0--  NOT T1SYNC: Timer1 External Clock Input Synchronization Control bit
	;					This bit is ignored. Timer1 uses the internal clock when TMR1CS = 0. 	
    ;         ------0-  TMR1CS: Timer1 Clock Source Select bit
	;						0 = Internal clock (FOSC/4) 	
    ;         -------0  TMR1ON: Timer1 On bit
	;						1 = Enables Timer1 
	;						0 = Stops Timer1 	
	movwf	T1CON

	; TMR1 Overflow Interrupt Flag not overflowed 	
	bcf    	PIR1,TMR1IF
	movlw  	high BLOADER_TIMER1
	movwf  	TMR1H		       
	movlw  	low BLOADER_TIMER1
	movwf  	TMR1L		       
	bsf    	T1CON,TMR1ON	; TIMER1 On

	call	BLoaderWaitForLogin
	movwf  	BLoaderQcode
	btfss	BLoaderQcode,LOGIN
	goto	BLoaderLogout

;------------------------------------------------------------
BLoaderLoop 				   
;----------

	; switch case BLoaderQcode
	;
	; case: WRITE INTO PROG MEM
	btfsc	BLoaderQcode,WRITE_PROG
	goto   	BLoaderWriteMem

	; case: WRITE INTO EEPROM
	btfsc	BLoaderQcode,WRITE_EEPROM
	goto   	BLoaderWriteMem		   

	; case: READ EEPROM
	btfsc	BLoaderQcode,READ_EEPROM
	goto   	BLoaderReadMem	

	; case: READ PROG
	btfsc	BLoaderQcode,READ_PROG
	goto   	BLoaderReadMem	

	; case: ERASE PROG
	btfsc	BLoaderQcode,ERASE_PROG
	goto   	BLoaderErasePROG

	; case: LOGIN
	btfsc	BLoaderQcode,LOGIN
	goto   	BLoaderLogin		 

	; case: LOGOUT
	btfsc	BLoaderQcode,LOGOUT
	goto	BLoaderLogout

BLoaderLoopRead
	call   	BLoaderReadByte	; Wait for a Byte from USART
	movwf  	BLoaderQcode
	goto   	BLoaderLoop

;------------
; LOGOUT
BLoaderLogout
	; Stop TIMER1
	BANKSEL	T1CON
	clrf  	T1CON			       
	; Reset USART Registers
	clrf  	RCSTA
	bsf   	STATUS,RP0
	clrf  	TXSTA			     
	bcf   	STATUS,RP0
	clrf  	PIR1
	;---------------------
	; Program User Execute
	clrf	PCLATH
	goto  	BLoaderExit 
	;---------------------

;------------------------------
; LOGIN acknowledgment
; Send start address of BLoader
; TX : Qhhllhhllhhllhhll0A
BLoaderLogin
	movfw	BLoaderQcode ; LOGIN acknowledgment
	call	BLoaderWriteByte

	movlw	high (BLOADER_ROOT & (PROG_MEM_SIZE-1))
	call	BLoaderWriteByte
	movlw	low (BLOADER_ROOT & (PROG_MEM_SIZE-1))
	call	BLoaderWriteByte

	movlw	high (BLOADER_SIZE)
	call	BLoaderWriteByte
	movlw	low (BLOADER_SIZE)
	call	BLoaderWriteByte

	movlw	high (BLOADER_OFFSET)
	call	BLoaderWriteByte
	movlw	low (BLOADER_OFFSET)
	call	BLoaderWriteByte
	movlw	high (BLOADER_VERSION)
	call	BLoaderWriteByte
	movlw	low (BLOADER_VERSION)

	goto	BLoaderWriteLine



;-------------------------
; ReadMem
; Send 2 bytes from Memory
; TX Format: Phhllhhll0A
BLoaderReadMem
	call 	BLoaderReadAddr	; Data Address HIGH,LOW
	call 	BLoaderSetEEADR
	btfss	BLoaderQcode,READ_EEPROM
	goto	BLoaderReadPROG

;-------------------------
; ReadEEPROM
; Read 2 bytes from EEPROM
BLoaderReadEEPROM
	BANKSEL	EECON1 
	bcf 	EECON1,EEPGD	; Point to Data memory	
	bsf		EECON1,RD		; read
	goto	BLoaderReadSend

;-----------------------
; ReadPROG
; Read 2 bytes from PROG
BLoaderReadPROG
	BANKSEL	EECON1 
	bsf 	EECON1,EEPGD	; Point to Data memory	
	bsf		EECON1,RD		; read
	nop 	; Any instructions here are ignored	
	nop		; Any instructions here are ignored	
	goto	BLoaderReadSend

;-----------------------
; Send 2 bytes from Mem
; TX : Paabb
BLoaderReadSend
	BANKSEL	EEDATH
	movf	EEDATH,W		
	movwf	BLoaderDataH
	movf	EEDATA,W		
	movwf	BLoaderDataL

	goto	BLoaderADFrameSend

;---------------------------------------------
BLoaderErasePROG
; RX Frame: Qaaaa = address of 32 words block
; TX Frame: Qaaaa
; The minimum erase block (row) is 32 words.	
; When initiating an erase sequence from the microcon- 	
; troller itself, a block of 32 words of program memory is 	
; erased. The Most Significant 11 bits of the 	
; EEADRH:EEADR point to the block being erased. 	
; EEADR< 4:0> are ignored	

	call 	BLoaderReadAddr	; Data Address HIGH,LOW
	call 	BLoaderSetEEADR

	BANKSEL EECON1 			; Select Bank of EECON1 	
	bsf 	EECON1,EEPGD 	; Point to PROGRAM memory 	
	bsf 	EECON1,WREN 	; Enable Write to memory 	
	bsf 	EECON1,FREE 	; Enable Row Erase operation 	

	; Erase Row Sequence
	movlw 	0x55
	movwf 	EECON2 		; Write 55h 	
	movlw 	0xAA 		
	movwf 	EECON2 		; Write AAh 	
	bsf 	EECON1,WR 	; Start Erase (CPU stall) 	
	nop 				; Any instructions here are ignored as processor 	
						; halts to begin Erase sequence 	
	nop 				; processor will stop here and wait for Erase complete 	
						; after Erase processor continues with 3rd instruction 	
	bcf 	EECON1,WREN	; Disable writes 	
	bcf 	INTCON,GIE 	; Enable interrupts (if using)	

	clrf	BLoaderDataH
	clrf	BLoaderDataL
	goto	BLoaderADFrameSend

;------------------------
; Common Write processing 
BLoaderWriteMem
	call 	BLoaderReadAddr	; Data Address HIGH+LOW+Checksum

	; Data byte HIGH
	call	BLoaderReadByte
	movwf	BLoaderDataH	; hold data into buffer
	addwf	BLoaderChkSum,f ;+ Checksum

	; Data byte LOW
	call	BLoaderReadByte
	movwf	BLoaderDataL	; hold data into buffer
	addwf	BLoaderChkSum,f ;+ Checksum

	; Checksum
	call	BLoaderReadByte 		
	xorwf	BLoaderChkSum,w	; chksum = w ?
	movlw	BLOADER_CHECKSUM_ERR
	btfss	STATUS,Z
	goto	BLoaderWriteLine; checksum error

	; checksum_ok, now we can write!

	;--------------------------------
	; (1) Load address into registers 
	;The EEADRH:EEADR register pair can address up to 	
	;a maximum of 256 bytes of data EEPROM, or up to a 	
	;maximum of 8K words of program EEPROM. When 	
	;selecting a data address value, only the LSB of the 	
	;address is written to the EEADR register. When select- 	
	;ing a program address value, the MSB of the address 	
	;is written to the EEADRH register and the LSB is 	
	;written to the EEADR register	

	;--------------
	; RP1	0 0 1 1
	; RP0	0 1	0 1
	;--------------
	; BANK	0 1 2 3
	;--------------
	call BLoaderSetEEADR

	;---------------------------------
	; (2) Load data into registers 
	;When interfacing the program memory block, the EEDATA 
	;and EEDATH registers form a two-byte word that 	
	;holds the 14-bit data for read/write and the EEADR and 	
	;EEADRH registers form a two-byte word that holds the 	
	;13-bit address of the EEPROM location being 	
	;accessed. The PIC16F87/88 devices have 4K words of 	
	;program Flash with an address range from 0000h to 	
	;0FFFh. Addresses above the range of the respective 	
	;device will wraparound to the beginning of program 	
	;memory.	
	BANKSEL	EEDATA
	movfw	BLoaderDataL
	movwf	EEDATA
	movfw	BLoaderDataH
	movwf	EEDATH

	;------------------------------------
	; (3) Split write into PROGRAM MEMORY
	btfss	BLoaderQcode,WRITE_EEPROM
	goto	BLoaderWritePROG

;-----------------
BLoaderWriteEEPROM
	; Write byte into EEPROM @Address
	BANKSEL	EECON1				
	; Set EECON1 data
    ;         0-------  EEPGD: Program/Data EEPROM Select bit 
	;						1 = Accesses program memory 
	;						0 = Accesses data memory	
    ;         -00-----  Unimplemented: Read as 0  
    ;         ---0----  FREE: EEPROM Forced Row Erase bit 
	;						1 = Erase the program memory row addressed by EEADRH:EEADR on the next WR command 	
	;						0 = Perform write only 	
    ;         ----0---  WRERR: EEPROM Error Flag bit 
	;						1 = A write operation is prematurely terminated (any MCLR or any WDT Reset during normal operation) 	
	;						0 = The write operation completed 	 	
    ;         -----0--  WREN: EEPROM Write Enable bit 
	;						1 = Allows write cycles 
	;						0 = Inhibits write to the EEPROM 	
    ;         ------0-  WR: Write Control
	;						1 = Initiates a write cycle. The bit is cleared by hardware once write is complete. The WR bit can only be set (not cleared) in software. 		
	;						0 = Write cycle to the EEPROM is complete 		
    ;         -------0  RD: Read Control bit 
	;						1 = Initiates an EEPROM read, RD is cleared in hardware. The RD bit can only be set (not cleared) in software. 	
	;						0 = Does not initiate an EEPROM read	

	bcf		EECON1,EEPGD	; WRITE into data mamory
	bsf		EECON1,WREN		; Allows write cycles	
	; Write Sequence
	movlw	0x55			
	movwf	EECON2			
	movlw	0xAA			
	movwf	EECON2			
	bsf		EECON1,WR		; Initiates a write cycle	
	
	; Wait for write end
	clrwdt					
	btfsc	EECON1,WR		; is done?
	goto	$-2				; no: loop wait
	bcf		EECON1,WREN		; yes: Inhibits write to the data EEPROM	

	; Write verify
	bsf		EECON1,RD		; read the data written
	BANKSEL	EEDATA
	movf	EEDATA,w
	movwf	BLoaderDataL	; and hold

	; Write OK
	goto	BLoaderADFrameSend

;--------------------------------------------------------
BLoaderWritePROG
; RX Frame: Qaaaadddd aaaa = address, dddd = data
; TX Frame: Qaaaarrrr aaaa = address, rrrr = data read)
	;--------------
	; RP1	0 0 1 1
	; RP0	0 1	0 1
	;--------------
	; BANK	0 1 2 3
	;--------------
	BANKSEL	EECON1
	bsf 	EECON1,EEPGD		; WRITE into program mamory
	bsf 	EECON1,WREN			; Allows write cycles
	bcf 	EECON1,FREE 		; Disable Row Erase operation 	
	movlw 	0x55				
	movwf 	EECON2				
	movlw	0xAA
	movwf 	EECON2				
	bsf 	EECON1,WR			; Initiates a write cycle	
	nop							; instructions here are ignored as processor	
	nop							; instructions here are ignored as processor	

	incf	BLoaderWord
	btfss   BLoaderWord,2
	; Ask for Write next word
	goto	BLoaderADFrameSend

	bcf 	EECON1,WREN			; Inhibits write
	clrf 	BLoaderWord

 IF 0
	;BANKSEL	EECON1
	bsf 	EECON1,EEPGD		; READ from program mamory
	bsf		EECON1,RD			; 
	nop							; instructions here are ignored as processor	
	nop							; instructions here are ignored as processor	
	BANKSEL	EEDATH
	movf	EEDATH,w			; Hold data re-read
	movwf	BLoaderDataH
	movf	EEDATA,w			
	movwf	BLoaderDataL
 ENDIF
	; Write OK
	goto	BLoaderADFrameSend

;--------------------------------- 	
; Send TX Write Frame Address,Data
BLoaderADFrameSend
	BANKSEL PIR1
	movfw	BLoaderQcode		
	call	BLoaderWriteByte 
	movfw	BLoaderAddrH
	call	BLoaderWriteByte 
	movfw	BLoaderAddrL
	call	BLoaderWriteByte 
	movfw	BLoaderDataH
	call	BLoaderWriteByte 
	movfw	BLoaderDataL

	goto	BLoaderWriteLine ;>>>>>>>>>>>>>>>>

;----------------------------------- 	
; Write one byte + new line to USART
; Input: WREG = Byte to send
BLoaderWriteLine
	;goto    BLoaderLoop ;<<<<<<<<<<<<<<<<<<<<<<<<<<
	; Send byte + end of line
	call	BLoaderWriteByte
	movlw	NEW_LINE
	call	BLoaderWriteByte

	goto	BLoaderLoopRead ; Next byte>>>>>>>>>>

;------------------------------
;----------SUB ROUTINE---------
;------------------------------

;-------------------------------------
; Read and Hold Data Address +Checksum
BLoaderReadAddr
	; Data Address HIGH
	BANKSEL	PIR1
	call	BLoaderReadByte
	movwf	BLoaderAddrH
	movwf	BLoaderChkSum	;init checksum 
	; Data Address LOW
	call	BLoaderReadByte
	movwf	BLoaderAddrL
	addwf	BLoaderChkSum,f	;+ Checksum
	return

;---------------------------
; Set EEADRH:EEADR registers
BLoaderSetEEADR
	; Data Address HIGH
	BANKSEL	EEADRH
	movfw	BLoaderAddrH
	movwf	EEADRH
	; Data Address LOW
	movfw	BLoaderAddrL
	movwf	EEADR
	return

;--------------------------- 	
; Write one Byte to USART 
; Input: WREG = Byte to send
BLoaderWriteByte
;TEST>>>>>>>>>>>>>>>>>>>>
	;return

	clrwdt
	btfss	PIR1,TXIF		       ; while(!TXIF)
	goto	$-2
	movwf	TXREG			       ; TXREG = octet
	return

;----------------------------- 	
; Read one Byte from USART 
; Output: WREG = Received Byte
BLoaderReadByte
;TEST<<<<<<<<<<<<<
	;return

	clrwdt
	btfss	PIR1,RCIF		       
	goto	$-2
	movf	RCREG,w	; Hold byte from RCREG into WREG
	return

;----------------------------------------------------- 	
; Wait for login from Host 
; Output: WREG = 'L' login or 'X' Execute user program
BLoaderWaitForLogin
	clrwdt
	btfss	PIR1,RCIF		       
	goto	$+3		; Nothing received: continue
	movf	RCREG,w ; Hold received byte RCREG into WREG
	return			; Exit OK >>>>>>>>>>>>>>

	btfss  	PIR1,TMR1IF	       		; TIMER1 overflow?
	goto   	BLoaderWaitForLogin 	; No: loop---------->
	bcf    	T1CON,TMR1ON	      	; Yes: TIMER1 Off
	decfsz 	BLoaderOVFtime,f		
	goto   	BLoaderWaitForLoginOVF	; Reset TIMER1------>
	;retlw 	'X'    	; Time out WREG=EXIT >>>>>>>>>>>>>>>
	retlw 	1<<LOGOUT  	; Time out WREG=EXIT >>>>>>>>>>>

	;-----------
	; Set TIMERH
BLoaderWaitForLoginOVF
	bcf    	PIR1,TMR1IF
	movlw  	high BLOADER_TIMER1
	movwf  	TMR1H		       
	bsf    	T1CON,TMR1ON			; TIMER1 On
BLoaderEnd
	goto	BLoaderWaitForLogin

;------------------------------
;--------END OF PROGRAM--------
;------------------------------

	END