﻿using chargePIC.Properties;
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.IO.Ports;
using System.Linq;
using System.Reflection;
using System.Resources;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace chargePIC
{
    public enum LogMsgType { Incoming, Outgoing, Normal, Warning, Error, TipMsg, Memory, HexFile };

    public partial class MainForm : Form
    {

        // Declare a Resource Manager instance.
        private ResourceManager LocRM = new ResourceManager("chargePIC.LocalStrings", typeof(MainForm).Assembly);
        private CultureInfo EnglishCulture = new CultureInfo("en-US");
        private CultureInfo FrenchCulture = new CultureInfo("fr-FR");

        // Various colors for log info
        private Color[] LogMsgTypeColor = { Color.Blue, Color.Green, Color.Black, Color.Orange, Color.Red, Color.Purple, Color.Black, Color.Navy };

        private Settings settings = Settings.Default;

        // Create an instance of the UART reference device
        private uartCommunication uartCom;

        System.IO.StreamWriter logFile = null;

        private int EEPROMstartAddr;
        private int config1Addr;
        private int config2Addr;

        private byte[] BLoaderRootAddr = new byte[2];
        private byte[] BLoaderSize = new byte[2];
        private byte[] BLoaderOffset = new byte[2];
        private byte[] BLoaderUserReset = new byte[2];
        private byte[] BLoaderVersion = new byte[2];

        public delegate void SetControl();
        public delegate void WriteLog(string m1, string m2);
        public delegate void StartProgres(int min, int max, int step);
        public delegate void StepProgres();
        public delegate void DispMem(bool p);
        public delegate void DispEEPROM();

        public MainForm()
        {
            //Load user settings before Initialize Component
            settings.Reload();

            // Default language
            if (settings.LanguageFrench)
                Thread.CurrentThread.CurrentUICulture = FrenchCulture;
            else
                Thread.CurrentThread.CurrentUICulture = EnglishCulture;

            // Create the UART reference Com object
            uartCom = new uartCommunication();

            InitializeComponent();

            // Add listener for UART events
            uartCom.portOpenEvent += new uartCommunication.uartEventsHandler(uartCom_portOpenEvent);
            uartCom.portOpenErrorEvent += new uartCommunication.uartEventsHandler(uartCom_portOpenErrorEvent);
            uartCom.parityListEvent += new uartCommunication.uartEventsHandler(uartCom_parityListEvent);
            uartCom.stopBitsListEvent += new uartCommunication.uartEventsHandler(uartCom_stopBitsListEvent);
            uartCom.portNameListEvent += new uartCommunication.uartEventsHandler(uartCom_portNameListEvent);

            uartCom.deviceNameListEvent += new uartCommunication.uartEventsHandler(uartCom_deviceNameListEvent);
            uartCom.devicePropertiesEvent += new uartCommunication.uartEventsHandler(uartCom_devicePropertiesEvent);

            uartCom.RxFrameErrorEvent += new uartCommunication.uartEventsHandler(uartCom_RxFrameErrorEvent);

            uartCom.TxLoginEvent += new uartCommunication.uartEventsHandler(uartCom_TxLoginEvent);
            uartCom.RxLoginEvent += new uartCommunication.uartEventsHandler(uartCom_RxLoginEvent);

            uartCom.startReadMemEvent += new uartCommunication.uartEventsHandler(uartCom_startReadMemEvent);
            uartCom.TxReadMemEvent += new uartCommunication.uartEventsHandler(uartCom_TxReadMemEvent);
            uartCom.RxReadMemEvent += new uartCommunication.uartEventsHandler(uartCom_RxReadMemEvent);
            uartCom.endReadMemEvent += new uartCommunication.uartEventsHandler(uartCom_endReadMemEvent);

            uartCom.startReadEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_startReadEEPROMEvent);
            uartCom.TxReadEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_TxReadEEPROMEvent);
            uartCom.RxReadEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_RxReadEEPROMEvent);
            uartCom.endReadEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_endReadEEPROMEvent);

            uartCom.startEraseMemEvent += new uartCommunication.uartEventsHandler(uartCom_startEraseMemEvent);
            uartCom.TxEraseMemEvent += new uartCommunication.uartEventsHandler(uartCom_TxEraseMemEvent);
            uartCom.RxEraseMemEvent += new uartCommunication.uartEventsHandler(uartCom_RxEraseMemEvent);
            uartCom.endEraseMemEvent += new uartCommunication.uartEventsHandler(uartCom_endEraseMemEvent);

            uartCom.startWriteMemEvent += new uartCommunication.uartEventsHandler(uartCom_startWriteMemEvent);
            uartCom.TxWriteMemEvent += new uartCommunication.uartEventsHandler(uartCom_TxWriteMemEvent);
            uartCom.RxWriteMemEvent += new uartCommunication.uartEventsHandler(uartCom_RxWriteMemEvent);
            uartCom.endWriteMemEvent += new uartCommunication.uartEventsHandler(uartCom_endWriteMemEvent);
            uartCom.startWriteMemVerifyEvent += new uartCommunication.uartEventsHandler(uartCom_startWriteMemVerifyEvent);
            uartCom.writeMemErrorEvent += new uartCommunication.uartEventsHandler(uartCom_writeMemErrorEvent);
            uartCom.endWriteMemVerifyEvent += new uartCommunication.uartEventsHandler(uartCom_endWriteMemVerifyEvent);

            uartCom.chkSumErrorEvent += new uartCommunication.uartEventsHandler(uartCom_chkSumErrorEvent);

            uartCom.startWriteEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_startWriteEEPROMEvent);
            uartCom.TxWriteEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_TxWriteEEPROMEvent);
            uartCom.RxWriteEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_RxWriteEEPROMEvent);
            uartCom.endWriteEEPROMEvent += new uartCommunication.uartEventsHandler(uartCom_endWriteEEPROMEvent);
            uartCom.startWriteEEPROMVerifyEvent += new uartCommunication.uartEventsHandler(uartCom_startWriteEEPROMVerifyEvent);
            uartCom.writeEEPROMErrorEvent += new uartCommunication.uartEventsHandler(uartCom_writeEEPROMErrorEvent);
            uartCom.endWriteEEPROMVerifyEvent += new uartCommunication.uartEventsHandler(uartCom_endWriteEEPROMVerifyEvent);

            uartCom.addMemListLineEvent += new uartCommunication.uartEventsHandler(uartCom_addMemListLineEvent);

            uartCom.addHexParseLineEvent += new uartCommunication.uartEventsHandler(uartCom_addHexParseLineEvent);
            uartCom.addHexLineEvent += new uartCommunication.uartEventsHandler(uartCom_addHexLineEvent);
            uartCom.hexConfig1Event += new uartCommunication.uartEventsHandler(uartCom_hexConfig1Event);
            uartCom.hexConfig2Event += new uartCommunication.uartEventsHandler(uartCom_hexConfig2Event);
            uartCom.endHexImportEvent += new uartCommunication.uartEventsHandler(uartCom_endHexImportEvent);
            uartCom.userHighAddrEvent += new uartCommunication.uartEventsHandler(uartCom_userHighAddrEvent);
            uartCom.userAppBoundaryErrorEvent += new uartCommunication.uartEventsHandler(uartCom_userAppBoundaryErrorEvent);

            uartCom.runUserAppEvent += new uartCommunication.uartEventsHandler(uartCom_runUserAppEvent);
        }

        private void uartCom_chkSumErrorEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Error, "CheckSumError", ": " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_userAppBoundaryErrorEvent(object sender, uartEventArgs e)
        {
            MessageBox.Show(LocRM.GetString("UserAppBoundaryError"), LocRM.GetString("WritingError"),
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private void uartCom_userHighAddrEvent(object sender, uartEventArgs e)
        {
            UserHighAddrDTextBox.Invoke(new EventHandler(delegate
            {
                UserHighAddrDTextBox.Text = string.Format("{0:D4}", e.Ptr);
            }));
            UserHighAddrTextBox.Invoke(new EventHandler(delegate
            {
                string str = string.Format("{0,4:X4}", e.Ptr);
                UserHighAddrTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);
            }));
        }

        private void uartCom_hexConfig2Event(object sender, uartEventArgs e)
        {
            config2TextBox.Invoke(new EventHandler(delegate
            {
                config2TextBox.Text = e.Str[0];
            }));
        }

        private void uartCom_hexConfig1Event(object sender, uartEventArgs e)
        {
            config1TextBox.Invoke(new EventHandler(delegate
            {
                config1TextBox.Text = e.Str[0];
            }));
        }

        private void uartCom_addHexLineEvent(object sender, uartEventArgs e)
        {
            if (HexFileTraceCheckBox.Checked)
                LogDisplay(LogMsgType.HexFile, "" + BitConverter.ToString(e.Buffer) + "\n");
            hexFileRichTextBox.SelectedText = string.Empty;
            hexFileRichTextBox.AppendText(e.Str[0]);
            hexFileRichTextBox.ScrollToCaret();
        }

        private void uartCom_addHexParseLineEvent(object sender, uartEventArgs e)
        {
            ListViewItem lvi = new ListViewItem(e.Str[0]);
            lvi.SubItems.Add(e.Str[1]);
            lvi.SubItems.Add(e.Str[2]);
            hexMachineListView.Items.Add(lvi);
        }

        private void uartCom_endHexImportEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, String.Format(LocRM.GetString("ReadHexFileCompletedAt") + " {0}\n", DateTime.Now));
            try
            {
                Invoke((SetControl)SetControlState);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_RxFrameErrorEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "RXFrameError" + ": " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_endReadMemEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "ReadMemoryCompletedAt", string.Format(" {0}\n", DateTime.Now));
            // Display Memory at end of Read Plan
            try
            {
                Invoke((DispMem)ViewMem, true);
            }
            catch (Exception ex) { return; }

            // Display EEPROM at end of Read Plan
            try
            {
                 Invoke((DispEEPROM)ViewEEPROM);
            }
            catch (Exception ex) { return; }

            try
            {
                Invoke((SetControl)SetControlState);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_RxReadMemEvent(object sender, uartEventArgs e)
        {
            if (settings.ReadMemTrace)
                LogDisplay(LogMsgType.Incoming, "Rx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_TxReadMemEvent(object sender, uartEventArgs e)
        {
            try
            {
                Invoke((StepProgres)StepProgressBar);
            }
            catch (Exception ex) { return; }

            if (settings.ReadMemTrace)
                LogDisplay(LogMsgType.Outgoing, "Tx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_startReadMemEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "ReadMemoryStartedAt", string.Format(" {0}\n", DateTime.Now));
            LogDisplay(LogMsgType.Normal, string.Format("M= [{0,4:X4} , {1,4:X4}]\n", e.Min, e.Max));

            try
            {
                Invoke((StartProgres)StartProgressBar, 0, e.Max - e.Min + 1, e.Step);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_endReadEEPROMEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "ReadEEPROMCompletedAt", string.Format(" {0}\n", DateTime.Now));

            // Display EEPROM at end of Read Plan
            try
            {
                Invoke((DispEEPROM)ViewEEPROM);
            }
            catch (Exception ex) { return; }
            try
            {
                Invoke((SetControl)SetControlState);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_RxReadEEPROMEvent(object sender, uartEventArgs e)
        {
            if (settings.ReadEEPROMtrace)
                LogDisplay(LogMsgType.Incoming, "Rx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_TxReadEEPROMEvent(object sender, uartEventArgs e)
        {
            try
            {
                Invoke((StepProgres)StepProgressBar);
            }
            catch (Exception ex) { return; }

            if (settings.ReadEEPROMtrace)
                LogDisplay(LogMsgType.Outgoing, "Tx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_startReadEEPROMEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "ReadEEPROMStartedAt", string.Format(" {0}\n", DateTime.Now));
            LogDisplay(LogMsgType.Normal, string.Format("M= [{0,4:X4} , {1,4:X4}]\n", e.Min, e.Max));

            try
            {
                Invoke((StartProgres)StartProgressBar, 0, e.Max - e.Min + 1, e.Step);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_startEraseMemEvent(object sender, uartEventArgs e)
        {

            LogDisplay(LogMsgType.Normal, "EraseMemoryStartedAt", string.Format(" {0}\n", DateTime.Now));
            LogDisplay(LogMsgType.Normal, string.Format("M= [{0,4:X4} , {1,4:X4}]\n", e.Min, e.Max));

            try
            {
                Invoke((StartProgres)StartProgressBar, 0, e.Max - e.Min + 1, e.Step);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_TxEraseMemEvent(object sender, uartEventArgs e)
        {
            try
            {
                Invoke((StepProgres)StepProgressBar);
            }
            catch (Exception ex) { return; }

            if (settings.EraseMemTrace)
                LogDisplay(LogMsgType.Outgoing, "Tx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_RxEraseMemEvent(object sender, uartEventArgs e)
        {
            if (settings.EraseMemTrace)
                LogDisplay(LogMsgType.Incoming, "Rx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_endEraseMemEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "EraseMemoryCompletedAt", string.Format(" {0}\n", DateTime.Now));
        }

        private void uartCom_startWriteMemEvent(object sender, uartEventArgs e)
        {

            LogDisplay(LogMsgType.Normal, "WriteMemoryStartedAt", string.Format(" {0}\n", DateTime.Now));
            LogDisplay(LogMsgType.Normal, string.Format("M= [{0,4:X4} , {1,4:X4}]\n", e.Min, e.Max));

            try
            {
                Invoke((StartProgres)StartProgressBar, 0, e.Max - e.Min + 1, e.Step);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_TxWriteMemEvent(object sender, uartEventArgs e)
        {

            try
            {
                Invoke((StepProgres)StepProgressBar);
            }
            catch (Exception ex) { return; }

            if (settings.WriteMemTrace)
                LogDisplay(LogMsgType.Outgoing, "Tx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_RxWriteMemEvent(object sender, uartEventArgs e)
        {
            if (settings.WriteMemTrace)
                LogDisplay(LogMsgType.Incoming, "Rx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_endWriteMemEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteMemoryCompletedAt", string.Format(" {0}\n", DateTime.Now));
        }

        private void uartCom_startWriteMemVerifyEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteVerifyStartedAt", string.Format(" {0}\n", DateTime.Now));
        }

        private void uartCom_writeMemErrorEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Error, "WriteErrorAt", string.Format(" [{0,4:X4}]={1} : {2}\n", e.Ptr, e.Str[0], e.Str[1]));
            MessageBox.Show(LocRM.GetString("WriteError"), LocRM.GetString("WritingError"),
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }


        private void uartCom_endWriteMemVerifyEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteVerifyCompletedAt", string.Format(" {0}\n", DateTime.Now));
            // Display Memory at end of Read Plan
            try
            {
                Invoke((DispMem)ViewMem, true);
            }
            catch (Exception ex) { return; }

            try
            {
                Invoke((SetControl)SetControlState);
            }
            catch (Exception ex) { return; }

            WriteEEPROMtoolStripButton_Click(sender, e);
        }

        private void uartCom_startWriteEEPROMEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteEEPROMStartedAt", string.Format(" {0}\n", DateTime.Now));
            LogDisplay(LogMsgType.Normal, string.Format("M= [{0,4:X4} , {1,4:X4}]\n", e.Min, e.Max));

            try
            {
                Invoke((StartProgres)StartProgressBar, 0, e.Max - e.Min + 1, e.Step);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_TxWriteEEPROMEvent(object sender, uartEventArgs e)
        {
            try
            {
                Invoke((StepProgres)StepProgressBar);
            }
            catch (Exception ex) { return; }

            if (settings.WriteEEPROMtrace)
                LogDisplay(LogMsgType.Outgoing, "Tx: " + BitConverter.ToString(e.Buffer) + "\n");

        }

        private void uartCom_RxWriteEEPROMEvent(object sender, uartEventArgs e)
        {
            if (settings.WriteEEPROMtrace)
                LogDisplay(LogMsgType.Incoming, "Rx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_endWriteEEPROMEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteEEPROMCompletedAt", string.Format(" {0}\n", DateTime.Now));
        }

        private void uartCom_startWriteEEPROMVerifyEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteEEPROMVerifyStartedAt", string.Format(" {0}\n", DateTime.Now));
        }

        private void uartCom_writeEEPROMErrorEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Error, "WriteEEPROMErrorAt", string.Format(" [{0,4:X4}]={1} : {2}\n", e.Ptr, e.Str[0], e.Str[1]));
            MessageBox.Show(LocRM.GetString("WriteError"), LocRM.GetString("WritingError"),
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private void uartCom_endWriteEEPROMVerifyEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "WriteEEPROMVerifyCompletedAt", string.Format(" {0}\n", DateTime.Now));

            // Display Memory at end of Read Plan
            try
            {
                Invoke((DispEEPROM)ViewEEPROM);
            }
            catch (Exception ex) { return; }
        }

        private void uartCom_addMemListLineEvent(object sender, uartEventArgs e)
        {
            ListViewItem lvi = new ListViewItem(e.Str[0]);
            lvi.UseItemStyleForSubItems = false;
            for(int j = 1; j<e.Str.Length-1; j++)
            {
                lvi.SubItems.Add(e.Str[j]);
                if (e.ProtectedArea) lvi.SubItems[j].ForeColor = Color.Red;
                else lvi.SubItems[j].ForeColor = Color.Blue;
            }
            lvi.SubItems.Add(e.Str[9]);
            if (e.EEPROM) EEPROMlistView.Items.Add(lvi);
            else progMemoryListView.Items.Add(lvi);
        }

        private void uartCom_runUserAppEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Outgoing, "Logout", ": " + BitConverter.ToString(e.Buffer) + "\n");
        }

        #region Progress Bar

        public void StartProgressBar(int min, int max, int step)
        {
            toolStripProgressBar1.Maximum = max;
            toolStripProgressBar1.Minimum = min;
            toolStripProgressBar1.Step = step;
        }

        public void StepProgressBar()
        {
            toolStripProgressBar1.PerformStep();
        }

        #endregion

        private void uartCom_RxLoginEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Incoming, "Rx: " + BitConverter.ToString(e.Buffer) + "\n");
 
            //Bootloader Address: rrrr
            BLoaderRootAddr[0] = e.Buffer[1];
            BLoaderRootAddr[1] = e.Buffer[2];
            BLoaderRootAddrTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderRootAddrTextBox.Text = BitConverter.ToString(BLoaderRootAddr);
            }));
            BLoaderRootAddrDTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderRootAddrDTextBox.Text = string.Format("{0}", (BLoaderRootAddr[0] * 16 * 16) + BLoaderRootAddr[1]);
            }));

            //Bootloader Size: ssss
            BLoaderSize[0] = e.Buffer[3];
            BLoaderSize[1] = e.Buffer[4];
            BLoaderSizeTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderSizeTextBox.Text = BitConverter.ToString(BLoaderSize);
            }));
            BLoaderSizeDTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderSizeDTextBox.Text = string.Format("{0}", (BLoaderSize[0] * 16 * 16) + BLoaderSize[1]);
            }));

            //Bootloader Offset: oooo
            BLoaderOffset[0] = e.Buffer[5];
            BLoaderOffset[1] = e.Buffer[6];
            BLoaderOffsetTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderOffsetTextBox.Text = BitConverter.ToString(BLoaderOffset);
            }));
            BLoaderOffsetDTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderOffsetDTextBox.Text = string.Format("{0}", (BLoaderOffset[0] * 16 * 16) + BLoaderOffset[1]);
            }));

            //Bootloader Version, revision: vvvv 
            BLoaderVersion[0] = e.Buffer[7];
            BLoaderVersion[1] = e.Buffer[8];
            BLoaderVersionTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderVersionTextBox.Text = BitConverter.ToString(BLoaderVersion).Replace("-", ".");
            }));

            //User Reset Address = Bootloader Root - offset
            int resetVector = int.Parse(BLoaderOffsetDTextBox.Text) + 0;
            BLoaderUserResetDTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderUserResetDTextBox.Text = string.Format("{0}", resetVector);
            }));
            byte[] b = BitConverter.GetBytes(resetVector);
            BLoaderUserReset[0] = b[1];
            BLoaderUserReset[1] = b[0];
            BLoaderUserResetTextBox.Invoke(new EventHandler(delegate
            {
                BLoaderUserResetTextBox.Text = BitConverter.ToString(BLoaderUserReset);
            }));

            LogDisplay(LogMsgType.Normal, "ConnectedToDeviceAt", string.Format(" {0}\n", DateTime.Now));

        }

        private void uartCom_TxLoginEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Outgoing, "Tx: " + BitConverter.ToString(e.Buffer) + "\n");
        }

        private void uartCom_portOpenErrorEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, LocRM.GetString("PortOpenError") + "\n");
            SetControlState();
        }

        private void uartCom_portOpenEvent(object sender, uartEventArgs e)
        {
            LogDisplay(LogMsgType.Normal, "PortOpened", "\n");
            SetControlState();
        }

        #region Settings

        /// <summary>
        /// Method to initialize serial port
        /// values to standard defaults
        /// </summary>
        private void SetDefaults()
        {
            langFrRadioButton.Checked = settings.LanguageFrench;
            langEnRadioButton.Checked = settings.LanguageEnglish;

            uartCom.Device_LoadNameList();

            uartCom.Parity_LoadList();
            uartCom.StopBits_LoadList();
            uartCom.PortName_LoadList();
            baudRateComboBox.Text = settings.BaudRate.ToString();
            dataBitsComboBox.Text = settings.DataBits.ToString();

            comTimer.Interval = Convert.ToInt32(settings.TimerInterval);
            timerIntervalNumericUpDown.Value = settings.TimerInterval;

            uartCom.RefreshComPortList(portNameToolStripComboBox.Items.Cast<string>(), portNameToolStripComboBox.SelectedItem);

            // If it is still available, select the last com port used
            if (portNameToolStripComboBox.Items.Contains(settings.PortName)) portNameToolStripComboBox.Text = settings.PortName;
            else if (portNameToolStripComboBox.Items.Count > 0)
            {
                portNameToolStripComboBox.SelectedIndex = portNameToolStripComboBox.Items.Count - 1;
                openPortButton_Click(this, new EventArgs());                
            }
            else
            {
                MessageBox.Show(this, LocRM.GetString("NoCOMportDetected"), LocRM.GetString("NoCOMportInstalled"), MessageBoxButtons.OK, MessageBoxIcon.Error);
                this.Close();
            }

            hexFilePathTextBox.Text = settings.HexFilePath;

            // Trace on Log
            HexFileTraceCheckBox.Checked = settings.HexFileTrace;
            ReadMemTraceCheckBox.Checked = settings.ReadMemTrace;
            EraseMemTraceCheckBox.Checked = settings.EraseMemTrace;
            WriteMemTraceCheckBox.Checked = settings.WriteMemTrace;
            ReadEEPROMtraceCheckBox.Checked = settings.ReadEEPROMtrace;
            WriteEEPROMtraceCheckBox.Checked = settings.WriteEEPROMtrace;
        }

        /// <summary> Save the user's settings. </summary>
        private void SaveSettings()
        {
            settings.BaudRate = int.Parse(baudRateComboBox.Text);
            settings.DataBits = int.Parse(dataBitsComboBox.Text);
            settings.Parity = (Parity)Enum.Parse(typeof(Parity), GetDefaultEnum(parityComboBox.Text, "Parity_", Enum.GetNames(typeof(Parity))));
            settings.StopBits = (StopBits)Enum.Parse(typeof(StopBits), GetDefaultEnum(stopBitsComboBox.Text, "StopBits_", Enum.GetNames(typeof(StopBits))));
            settings.PortName = portNameToolStripComboBox.Text;

            settings.TimerInterval = timerIntervalNumericUpDown.Value;
            settings.LanguageFrench = langFrRadioButton.Checked;
            settings.LanguageEnglish = langEnRadioButton.Checked;

            settings.HexFilePath = hexFilePathTextBox.Text;

            settings.HexFileTrace = HexFileTraceCheckBox.Checked;
            settings.ReadMemTrace = ReadMemTraceCheckBox.Checked;
            settings.EraseMemTrace = EraseMemTraceCheckBox.Checked;
            settings.WriteMemTrace = WriteMemTraceCheckBox.Checked;
            settings.ReadEEPROMtrace = ReadEEPROMtraceCheckBox.Checked;
            settings.WriteEEPROMtrace = WriteEEPROMtraceCheckBox.Checked;

            settings.Save();
        }

        #endregion
        private void SetControlState()
        {
            LoginToolStripButton.Enabled = (uartCom.PortOpen && !uartCom.Login);
            RunToolStripButton.Enabled = (uartCom.PortOpen && uartCom.Login);
            ReadToolStripButton.Enabled = (uartCom.PortOpen && uartCom.Login);
            WriteMemToolStripButton.Enabled = (uartCom.PortOpen && uartCom.Login && uartCom.Hex & !uartCom.HexError);
            ReadEEPROMtoolStripButton.Enabled = (uartCom.PortOpen && uartCom.Login);
            WriteEEPROMtoolStripButton.Enabled = (uartCom.PortOpen && uartCom.Login && uartCom.Hex & !uartCom.HexError);

            loginToolStripMenuItem.Enabled = LoginToolStripButton.Enabled;
            writeToolStripMenuItem.Enabled = WriteMemToolStripButton.Enabled;
            readToolStripMenuItem.Enabled = ReadToolStripButton.Enabled;
            writeEEPROMToolStripMenuItem.Enabled = WriteEEPROMtoolStripButton.Enabled;
            readEEPROMToolStripMenuItem.Enabled = ReadEEPROMtoolStripButton.Enabled;
            runToolStripMenuItem.Enabled = RunToolStripButton.Enabled;
        }

        #region Log

        private void ClearLog()
        {
            logRichTextBox.Clear();
        }

        public void WriteLogFile(string locMsg, string msg)
        {
            logFile.WriteLine(LocRM.GetString(locMsg) + msg);
        }

        /// <summary>
        /// method to display the data to & from the port
        /// on the screen
        /// </summary>
        /// <param name="type">MessageType of the message</param>
        /// <param name="msg">Message to display</param>
        private void LogDisplay(LogMsgType type, string locMsg, string msg)
        {
            Invoke((WriteLog)WriteLogFile, locMsg, msg);

            logRichTextBox.Invoke(new EventHandler(delegate
            {
                logRichTextBox.SelectedText = string.Empty;
                //_log.SelectionFont = new Font(_log.SelectionFont, FontStyle.Bold);
                logRichTextBox.SelectionFont = new Font(logRichTextBox.SelectionFont, FontStyle.Regular);
                logRichTextBox.SelectionColor = LogMsgTypeColor[(int)type];
                logRichTextBox.AppendText(LocRM.GetString(locMsg) + msg);
                logRichTextBox.ScrollToCaret();
            }));
        }

        /// <summary>
        /// method to display the data to & from the port
        /// on the screen
        /// </summary>
        /// <param name="type">MessageType of the message</param>
        /// <param name="msg">Message to display</param>
        private void LogDisplay(LogMsgType type, string msg)
        {
            logFile.WriteLine(msg);

            logRichTextBox.Invoke(new EventHandler(delegate
            {
                logRichTextBox.SelectedText = string.Empty;
                //_log.SelectionFont = new Font(_log.SelectionFont, FontStyle.Bold);
                logRichTextBox.SelectionFont = new Font(logRichTextBox.SelectionFont, FontStyle.Regular);
                logRichTextBox.SelectionColor = LogMsgTypeColor[(int)type];
                logRichTextBox.AppendText(msg);
                logRichTextBox.ScrollToCaret();
            }));
        }
        #endregion

        private void uartCom_addLoglineEvent(object sender, uartEventArgs e)
        {
            throw new NotImplementedException();
        }

        private void uartCom_devicePropertiesEvent(object sender, uartEventArgs e)
        {
            DeviceIDTextBox.Text = e.Dev.ID;
            ProgMemSizeDTextBox.Text = e.Dev.ProgMemSize.HexToInt().ToString();
            string str = string.Format("{0,4:X4}", int.Parse(ProgMemSizeDTextBox.Text));
            ProgMemSizeTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);

            EEPROMsizeDTextBox.Text = e.Dev.EEPROMsize.HexToInt().ToString();
            str = string.Format("{0,4:X4}", int.Parse(EEPROMsizeDTextBox.Text));
            EEPROMsizeTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);

            EEPROMstartAddr = e.Dev.EEPROMstartAddr.HexToInt();
            EEPROMstartAddrDTextBox.Text = EEPROMstartAddr.ToString();
            str = string.Format("{0,4:X4}", EEPROMstartAddr);
            EEPROMstartAddrTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);

            config1Addr = e.Dev.Config1Addr.HexToInt();
            str = string.Format("{0,4:X4}", config1Addr);
            config1AddrTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);

            config2Addr = e.Dev.Config2Addr.HexToInt();
            str = string.Format("{0,4:X4}", config2Addr);
            config2AddrTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);
        }

        private void uartCom_deviceNameListEvent(object sender, uartEventArgs e)
        {
            DeviceToolStripComboBox.Items.Clear();
            DeviceToolStripComboBox.Items.AddRange(e.Obj);
            DeviceToolStripComboBox.Text = settings.DeviceName;
        }

        private void uartCom_parityListEvent(object sender, uartEventArgs e)
        {
            parityComboBox.Items.Clear();
            parityComboBox.Items.AddRange(GetLocalizedEnum("Parity_", e.Str));
            parityComboBox.Text = LocRM.GetString("Parity_" + settings.Parity.ToString());
        }

        private void uartCom_stopBitsListEvent(object sender, uartEventArgs e)
        {
            stopBitsComboBox.Items.Clear();
            stopBitsComboBox.Items.AddRange(GetLocalizedEnum("StopBits_", e.Str));
            stopBitsComboBox.Text = LocRM.GetString("StopBits_" + settings.StopBits.ToString());
        }

        private void uartCom_portNameListEvent(object sender, uartEventArgs e)
        {
            portNameToolStripComboBox.Items.Clear();
            portNameToolStripComboBox.Items.AddRange(e.Str);
            portNameToolStripComboBox.Text = settings.PortName;
        }

        /// <summary> Get localized string array from ressource. </summary>
        /// <param name="prefix"> prefix = "Parity_" or "StopBits_". </param>
        /// <param name="def"> The default Enum string array. </param>
        private string[] GetLocalizedEnum(string prefix, string[] def)
        {
            string[] localArray = new string[def.Length];
            def.CopyTo(localArray, 0);
            int i = 0;
            foreach (string str in def)
            {
                localArray[i++] = LocRM.GetString(prefix + str);
            }

            return localArray;
        }

        /// <summary> Get default string from ressource. </summary>
        /// <param name="prefix"> prefix = "Parity_" or "StopBits_". </param>
        /// <param name="def"> The default Enum string array. </param>
        /// <param name="locVal"> The value to find in ressource. </param>
        private string GetDefaultEnum(string locVal, string prefix, string[] def)
        {
            int i = 0;
            string val = def[0]; // if not found
            foreach (string str in def)
            {
                // Find local value with key = prefix+str
                if (LocRM.GetString(prefix + str).Equals(locVal))
                {
                    val = def[i]; // return dafault value
                    break;
                }
                i++;
            }
            return val;
        }

        private void MainForm_Shown(object sender, EventArgs e)
        {
            SetDefaults();

            // After initialize, select the Log tab and open Log File.
            tabControl2.SelectTab("LogTabPage");
            tabControl2.SelectedTab.Text = "chargePICLog.txt";

            logFile = new System.IO.StreamWriter(@tabControl2.SelectedTab.Text, false);

            ClearLog();

            LogDisplay(LogMsgType.Normal, String.Format(LocRM.GetString("ApplicationStartedAt") + " {0}\n", DateTime.Now));

            openPortButton_Click(sender, e);

            if (File.Exists(@hexFilePathTextBox.Text))
                hexFileViewButton_Click(sender, e);
        }

        private void DeviceToolStripComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            uartCom.Device_LoadProperties(DeviceToolStripComboBox.Text);
        }

        private void openPortButton_Click(object sender, EventArgs e)
        {
            uartCom.OpenPort(baudRateComboBox.Text, dataBitsComboBox.Text,
                GetDefaultEnum(stopBitsComboBox.Text, "StopBits_", Enum.GetNames(typeof(StopBits))),
                GetDefaultEnum(parityComboBox.Text, "Parity_", Enum.GetNames(typeof(Parity))), portNameToolStripComboBox.Text);
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            // The form is closing, save the user's preferences
            SaveSettings();
            logFile.Close();

        }

        private void ClearLogToolStripButton_Click(object sender, EventArgs e)
        {
            ClearLog();

        }

        private void clearLogToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ClearLogToolStripButton_Click(sender, e);
        }

        private void LoginToolStripButton_Click(object sender, EventArgs e)
        {
            LogDisplay(LogMsgType.Normal, String.Format(LocRM.GetString("ConnectToDeviceStartedAt") + " {0}\n", DateTime.Now));
            // Send the binary data out the port
            uartCom.TxLogin();
        }

        private void loginToolStripMenuItem_Click(object sender, EventArgs e)
        {
            LoginToolStripButton_Click(sender, e);
        }

        private void hexFileViewButton_Click(object sender, EventArgs e)
        {
            hexFileRichTextBox.Clear();
            hexMachineListView.Items.Clear(); //Clear view

            LogDisplay(LogMsgType.Normal, String.Format(LocRM.GetString("ReadHexFileStartedAt") + " {0}\n", DateTime.Now));
            uartCom.ImportHexFile(hexFilePathTextBox.Text);

            ProgMemViewButton_Click(sender, e);
            EEPROMviewButton_Click(sender, e);
        }

        private void ImportHexFileToolStripButton_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFileDialog1 = new OpenFileDialog();

            openFileDialog1.InitialDirectory = settings.HexFilePath;
            openFileDialog1.Filter = "hex files (*.hex)|*.hex|All files (*.*)|*.*";
            openFileDialog1.FilterIndex = 1;
            openFileDialog1.RestoreDirectory = true;
            openFileDialog1.CheckFileExists = true;
            openFileDialog1.CheckPathExists = true;
            openFileDialog1.ReadOnlyChecked = true;
            openFileDialog1.ShowReadOnly = true;
            if (openFileDialog1.ShowDialog() == DialogResult.OK)
            {
                hexFilePathTextBox.Text = openFileDialog1.FileName;
                hexFileViewButton_Click(sender, e);
            }

        }

        private void importHexFileToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ImportHexFileToolStripButton_Click(sender, e);
        }

        private void saveSettingButton_Click(object sender, EventArgs e)
        {
            SaveSettings();
        }

        private void ProgMemViewButton_Click(object sender, EventArgs e)
        {
            ViewMem(false);
        }

        private void EEPROMviewButton_Click(object sender, EventArgs e)
        {
            ViewEEPROM();
        }

        private void ViewMem(bool protectedArea)
        {
            progMemoryListView.Items.Clear();
            uartCom.ViewProgMem(protectedArea);
        }

        private void ViewMem(int x, int y, int z)
        {
            progMemoryListView.Items.Clear();
            uartCom.ViewProgMem(x, y, z);
        }

        private void ViewEEPROM()
        {
            EEPROMlistView.Items.Clear();
            uartCom.ViewEEPROM();
        }

        private void ViewEEPROM(int x, int y, int z)
        {
            EEPROMlistView.Items.Clear();
            uartCom.ViewEEPROM(x, y, z);
        }

        private void WriteEEPROMtoolStripButton_Click(object sender, EventArgs e)
        {
           uartCom.UploadUserData();
        }

        private void writeEEPROMToolStripMenuItem_Click(object sender, EventArgs e)
        {
            WriteEEPROMtoolStripButton_Click(sender, e);
        }

        private void ReadEEPROMtoolStripButton_Click(object sender, EventArgs e)
        {
            uartCom.DownLoadEEPROM();
        }

        private void readEEPROMToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ReadEEPROMtoolStripButton_Click(sender, e);
        }

        private void WriteMemToolStripButton_Click(object sender, EventArgs e)
        {
            uartCom.UpLoadUserApp();
        }

        private void writeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            WriteMemToolStripButton_Click(sender, e);
        }

        private void RunToolStripButton_Click(object sender, EventArgs e)
        {
            uartCom.RunUserApp();
        }

        private void runToolStripMenuItem_Click(object sender, EventArgs e)
        {
            RunToolStripButton_Click(sender, e);
        }

        private void ReadToolStripButton_Click(object sender, EventArgs e)
        {
            uartCom.DownLoadMemory();
        }

        private void readToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ReadToolStripButton_Click(sender, e);
        }

        private void comTimer_Tick(object sender, EventArgs e)
        {
            uartCom.RefreshComPortList(portNameToolStripComboBox.Items.Cast<string>(), portNameToolStripComboBox.SelectedItem);
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();

        }

        private void dataBitsComboBox_Validating(object sender, CancelEventArgs e)
        {
            int x; e.Cancel = !int.TryParse(dataBitsComboBox.Text, out x);
        }

        private void baudRateComboBox_Validating(object sender, CancelEventArgs e)
        {
            int x; e.Cancel = !int.TryParse(baudRateComboBox.Text, out x);

        }

        private void AboutToolStripButton_Click(object sender, EventArgs e)
        {
            // Show the user the about dialog
            (new AboutForm()).ShowDialog(this);

        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            AboutToolStripButton_Click(sender, e);

        }

        private void baudRateCalculatorToolStripMenuItem_Click(object sender, EventArgs e)
        {
            (new BaudRateCalculForm()).ShowDialog(this);

        }

        private void timeOutCalculatorToolStripMenuItem_Click(object sender, EventArgs e)
        {
            (new TimeOutCalculForm()).ShowDialog(this);

        }
    }
}
