﻿using System;
using System.Collections.Generic;
using System.IO.Ports;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace chargePIC
{
    public enum DeviceName { PIC16F87, PIC16F88 };
    public struct Device
    {
        private string name;
        private string iD;
        private string progMemSize;
        private string eEPROMsize;
        private string eEPROMstartAddr;
        private string config1Addr;
        private string config2Addr;
 
        public Device(string name, string id, string progMemSize, string EEPROMsize, string EEPROMstartAddr, string config1Addr, string config2Addr)
        {
            this.name = name;
            this.iD = id;
            this.progMemSize = progMemSize;
            this.eEPROMsize = EEPROMsize;
            this.eEPROMstartAddr = EEPROMstartAddr;
            this.config1Addr = config1Addr;
            this.config2Addr = config2Addr;
        }

        public string Name { get { return name; } }
        public string ID { get { return iD; } }
        public string ProgMemSize { get { return progMemSize; } }
        public string EEPROMsize { get { return eEPROMsize; } }
        public string EEPROMstartAddr { get { return eEPROMstartAddr; } }
        public string Config1Addr { get { return config1Addr; } }
        public string Config2Addr { get { return config2Addr; } }
    }

    class uartCommunication
    {
        #region Properties

        /// <summary>
        /// Property to hold the BaudRate
        /// </summary>
        public string BaudRate { get; set; }
        public string Parity { get; set; }
        public string StopBits { get; set; }
        public string DataBits { get; set; }
        public string PortName { get; set; }
        public bool IsOpen { get; set; }
        public bool PortOpen { get; private set; }
        public bool Login { get; private set; }
        public bool Verify { get; private set; }
        public bool Read { get; private set; }
        public int config1Addr { get; private set; }
        public int config2Addr { get; private set; }
        public int EEPROMstartAddr { get; private set; }
        public bool Hex { get; private set; }
        public bool HexError { get; private set; }
        public int BLoaderRootAddr { get; private set; }
        public int BLoaderSize { get; private set; }
        public int BLoaderOffset { get; private set; }
        public int UserHighAddr { get; private set; }
        public bool Erase { get; private set; }
        public byte[] LastTx { get; private set; }
        public bool Write { get; private set; }
        public int EEPROMsize { get; private set; }

        #endregion

        #region Variables
        private SettingsDevice settings = SettingsDevice.Default;

        private SerialPort comPort = new SerialPort();
        private Queue<byte> receivedData = new Queue<byte>();

        private Memory progMem;
        private Memory dataEEPROM;
        private Word[] CopyProg;
        private Word[] CopyData;

        #endregion

        #region Constants
        // Communication protocol
        internal const byte LOGIN_TO_DEVICE = 0x01; //RX TX Qhhllhhllhhllhhll
        internal const byte WRITE_TO_PROG = 0x02; //RX TX
        internal const byte WRITE_TO_EEPROM = 0x04; //RX TX
        internal const byte READ_PROG = 0x08; //RX TX
        internal const byte READ_EEPROM = 0x10; //RX TX
        internal const byte ERASE_PROG = 0x20; //RX TX
        internal const byte LOGOUT_OF_DEVICE = 0x80; //TX
        internal const byte CHECKSUM_ERROR = (byte)'?'; //RX
        #endregion

        /// <summary>
        /// Create a delegate for the UART event handler
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        //public delegate void uartEventsHandler(object sender, EventArgs e);

        public delegate void uartEventsHandler(object sender, uartEventArgs e);

        /// <summary>
        /// Define the Login event
        /// </summary>
        public event uartEventsHandler uartLoginEvent;

        /// <summary>
        /// Define the Parity List event
        /// </summary>
        public event uartEventsHandler parityListEvent;

        /// <summary>
        /// Define the Stop Bits List event
        /// </summary>
        public event uartEventsHandler stopBitsListEvent;

        /// <summary>
        /// Define the Port Name List event
        /// </summary>
        public event uartEventsHandler portNameListEvent;

        /// <summary>
        /// Define the Device Name List event
        /// </summary>
        public event uartEventsHandler deviceNameListEvent;

        /// <summary>
        /// Define the Device Properties event
        /// </summary>
        public event uartEventsHandler devicePropertiesEvent;

        /// <summary>
        /// Define the Port Open event
        /// </summary>
        public event uartEventsHandler portOpenEvent;

        /// <summary>
        /// Define the add Hex File Parse Line event
        /// </summary>
        public event uartEventsHandler addHexParseLineEvent;

        /// <summary>
        /// Define the add Hex File Line event
        /// </summary>
        public event uartEventsHandler addHexLineEvent;

        /// <summary>
        /// Define the hex file Config1 value event
        /// </summary>
        public event uartEventsHandler hexConfig1Event;

        /// <summary>
        /// Define the hex file Config2 value event
        /// </summary>
        public event uartEventsHandler hexConfig2Event;

        /// <summary>
        /// Define the end of hex file event
        /// </summary>
        public event uartEventsHandler endHexImportEvent;

        /// <summary>
        /// Define the high user application address value event
        /// </summary>
        public event uartEventsHandler userHighAddrEvent;

        /// <summary>
        /// Define the  user application Boundary Error event
        /// </summary>
        public event uartEventsHandler userAppBoundaryErrorEvent;

        /// <summary>
        /// Define the Port Open Error event
        /// </summary>
        public event uartEventsHandler portOpenErrorEvent;

        /// <summary>
        /// Define the Receive Login event
        /// </summary>
        public event uartEventsHandler RxLoginEvent;

        /// <summary>
        /// Define the Transmit Login event
        /// </summary>
        public event uartEventsHandler TxLoginEvent;

        /// <summary>
        /// Define the Receive Frame Error event
        /// </summary>
        public event uartEventsHandler RxFrameErrorEvent;

        /// <summary>
        /// Define the Start Read Memory event
        /// </summary>
        public event uartEventsHandler startReadMemEvent;

        /// <summary>
        /// Define the Receive Read Memory event
        /// </summary>
        public event uartEventsHandler RxReadMemEvent;

        /// <summary>
        /// Define the Transmit Read Memory event
        /// </summary>
        public event uartEventsHandler TxReadMemEvent;

        /// <summary>
        /// Define the End of Read Memory event
        /// </summary>
        public event uartEventsHandler endReadMemEvent;

        /// <summary>
        /// Define the Start Read EEPROM event
        /// </summary>
        public event uartEventsHandler startReadEEPROMEvent;

        /// <summary>
        /// Define the Receive Read EEPROM event
        /// </summary>
        public event uartEventsHandler RxReadEEPROMEvent;

        /// <summary>
        /// Define the Transmit Read EEPROM event
        /// </summary>
        public event uartEventsHandler TxReadEEPROMEvent;

        /// <summary>
        /// Define the End of Read EEPROM event
        /// </summary>
        public event uartEventsHandler endReadEEPROMEvent;

        /// <summary>
        /// Define the Start Erase Memory event
        /// </summary>
        public event uartEventsHandler startEraseMemEvent;

        /// <summary>
        /// Define the Receive Erase Memory event
        /// </summary>
        public event uartEventsHandler RxEraseMemEvent;

        /// <summary>
        /// Define the Transmit Erase Memory event
        /// </summary>
        public event uartEventsHandler TxEraseMemEvent;

        /// <summary>
        /// Define the End of Erase Memory event
        /// </summary>
        public event uartEventsHandler endEraseMemEvent;

        /// <summary>
        /// Define the Start Write Memory event
        /// </summary>
        public event uartEventsHandler startWriteMemEvent;

        /// <summary>
        /// Define the Receive Write Memory event
        /// </summary>
        public event uartEventsHandler RxWriteMemEvent;

        /// <summary>
        /// Define the Transmit Write Memory event
        /// </summary>
        public event uartEventsHandler TxWriteMemEvent;

        /// <summary>
        /// Define the End of Write Memory event
        /// </summary>
        public event uartEventsHandler endWriteMemEvent;

        /// <summary>
        /// Define the Start Verify Write Memory event
        /// </summary>
        public event uartEventsHandler startWriteMemVerifyEvent;

        /// <summary>
        /// Define the Write Memory Error event
        /// </summary>
        public event uartEventsHandler writeMemErrorEvent;

        /// <summary>
        /// Define the End of Verify Write Memory event
        /// </summary>
        public event uartEventsHandler endWriteMemVerifyEvent;

        /// <summary>
        /// Define the Check Sum Error event
        /// </summary>
        public event uartEventsHandler chkSumErrorEvent;

        /// <summary>
        /// Define the Start Write EEPROM event
        /// </summary>
        public event uartEventsHandler startWriteEEPROMEvent;

        /// <summary>
        /// Define the Receive Write EEPROM event
        /// </summary>
        public event uartEventsHandler RxWriteEEPROMEvent;

        /// <summary>
        /// Define the Transmit Write EEPROM event
        /// </summary>
        public event uartEventsHandler TxWriteEEPROMEvent;

        /// <summary>
        /// Define the End of Write EEPROM event
        /// </summary>
        public event uartEventsHandler endWriteEEPROMEvent;

        /// <summary>
        /// Define the Start Verify Write EEPROM event
        /// </summary>
        public event uartEventsHandler startWriteEEPROMVerifyEvent;

        /// <summary>
        /// Define the Write EEPROM Error event
        /// </summary>
        public event uartEventsHandler writeEEPROMErrorEvent;

        /// <summary>
        /// Define the End of Verify Write EEPROM event
        /// </summary>
        public event uartEventsHandler endWriteEEPROMVerifyEvent;

        /// <summary>
        /// Define the Run User Application event
        /// </summary>
        public event uartEventsHandler runUserAppEvent;

        /// <summary>
        /// Define the Add line to memory list event
        /// </summary>
        public event uartEventsHandler addMemListLineEvent;

        #region Constructors

        /// <summary>
        /// Comstructor to set the properties of our
        /// serial port communicator to nothing
        /// </summary>
        public uartCommunication()
        {
            Hex = PortOpen = Login = false;
            // Add a listener for Serialport events
            comPort.DataReceived += new SerialDataReceivedEventHandler(comPort_DataReceived);
        }
        #endregion

        /// <summary>
        /// Parity List event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onParityListEvent(uartEventArgs e)
        {
            if (parityListEvent != null)
            {
                parityListEvent(this, e);
            }
        }

        /// <summary>
        /// Stop Bits List event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStopBitsListEvent(uartEventArgs e)
        {
            if (stopBitsListEvent != null)
            {
                stopBitsListEvent(this, e);
            }
        }

        /// <summary>
        /// Port name list event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onPortNameListEvent(uartEventArgs e)
        {
            if (portNameListEvent != null)
            {
                portNameListEvent(this, e);
            }
        }

        /// <summary>
        /// Device name list event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onDeviceNameListEvent(uartEventArgs e)
        {
            if (deviceNameListEvent != null)
            {
                deviceNameListEvent(this, e);
            }
        }

        /// <summary>
        /// Device Properties event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onDevicePropertiesEvent(uartEventArgs e)
        {
            if (devicePropertiesEvent != null)
            {
                devicePropertiesEvent(this, e);
            }
        }

        /// <summary>
        /// Port Open event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onPortOpenEvent(uartEventArgs e)
        {
            if (portOpenEvent != null)
            {
                portOpenEvent(this, e);
            }
        }

        /// <summary>
        /// Port Open Error event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onPortOpenErrorEvent(uartEventArgs e)
        {
            if (portOpenErrorEvent != null)
            {
                portOpenErrorEvent(this, e);
            }
        }

        /// <summary>
        /// Transmit Login event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onTxLoginEvent(uartEventArgs e)
        {
            if (TxLoginEvent != null)
            {
                TxLoginEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Login event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxLoginEvent(uartEventArgs e)
        {
            if (RxLoginEvent != null)
            {
                RxLoginEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Frame Error event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxFrameErrorEvent(uartEventArgs e)
        {
            if (RxFrameErrorEvent != null)
            {
                RxFrameErrorEvent(this, e);
            }
        }

        /// <summary>
        /// Start Read Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartReadMemEvent(uartEventArgs e)
        {
            if (startReadMemEvent != null)
            {
                startReadMemEvent(this, e);
            }
        }

        /// <summary>
        /// Transmit Read Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onTxReadMemEvent(uartEventArgs e)
        {
            if (TxReadMemEvent != null)
            {
                TxReadMemEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Read Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxReadMemEvent(uartEventArgs e)
        {
            if (RxReadMemEvent != null)
            {
                RxReadMemEvent(this, e);
            }
        }

        /// <summary>
        /// End of Read Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndReadMemEvent(uartEventArgs e)
        {
            if (endReadMemEvent != null)
            {
                endReadMemEvent(this, e);
            }
        }

        /// <summary>
        /// Start Read EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartReadEEPROMEvent(uartEventArgs e)
        {
            if (startReadEEPROMEvent != null)
            {
                startReadEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// Transmit Read EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onTxReadEEPROMEvent(uartEventArgs e)
        {
            if (TxReadEEPROMEvent != null)
            {
                TxReadEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Read EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxReadEEPROMEvent(uartEventArgs e)
        {
            if (RxReadEEPROMEvent != null)
            {
                RxReadEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// End of Read EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndReadEEPROMEvent(uartEventArgs e)
        {
            if (endReadEEPROMEvent != null)
            {
                endReadEEPROMEvent(this, e);
            }
        }
        /// <summary>
        /// Start Erase Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartEraseMemEvent(uartEventArgs e)
        {
            if (startEraseMemEvent != null)
            {
                startEraseMemEvent(this, e);
            }
        }

        /// <summary>
        /// Transmit Erase Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onTxEraseMemEvent(uartEventArgs e)
        {
            if (TxEraseMemEvent != null)
            {
                TxEraseMemEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Erase Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxEraseMemEvent(uartEventArgs e)
        {
            if (RxEraseMemEvent != null)
            {
                RxEraseMemEvent(this, e);
            }
        }

        /// <summary>
        /// End of Erase Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndEraseMemEvent(uartEventArgs e)
        {
            if (endEraseMemEvent != null)
            {
                endEraseMemEvent(this, e);
            }
        }

        /// <summary>
        /// Start Write Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartWriteMemEvent(uartEventArgs e)
        {
            if (startWriteMemEvent != null)
            {
                startWriteMemEvent(this, e);
            }
        }

        /// <summary>
        /// Transmit Write Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onTxWriteMemEvent(uartEventArgs e)
        {
            if (TxWriteMemEvent != null)
            {
                TxWriteMemEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Write Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxWriteMemEvent(uartEventArgs e)
        {
            if (RxWriteMemEvent != null)
            {
                RxWriteMemEvent(this, e);
            }
        }

        /// <summary>
        /// End of Write Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndWriteMemEvent(uartEventArgs e)
        {
            if (endWriteMemEvent != null)
            {
                endWriteMemEvent(this, e);
            }
        }

        /// <summary>
        /// Start Verify Write Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartWriteMemVerifyEvent(uartEventArgs e)
        {
            if (startWriteMemVerifyEvent != null)
            {
                startWriteMemVerifyEvent(this, e);
            }
        }

        /// <summary>
        /// Verify Write Memory Error event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onWriteMemErrorEvent(uartEventArgs e)
        {
            if (writeMemErrorEvent != null)
            {
                writeMemErrorEvent(this, e);
            }
        }

        /// <summary>
        /// End of Verify Write Memory event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndWriteMemVerifyEvent(uartEventArgs e)
        {
            if (endWriteMemVerifyEvent != null)
            {
                endWriteMemVerifyEvent(this, e);
            }
        }

        /// <summary>
        /// Check Sum Error event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onChkSumErrorEvent(uartEventArgs e)
        {
            if (chkSumErrorEvent != null)
            {
                chkSumErrorEvent(this, e);
            }
        }

        /// <summary>
        /// Start Write EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartWriteEEPROMEvent(uartEventArgs e)
        {
            if (startWriteEEPROMEvent != null)
            {
                startWriteEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// Transmit Write EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onTxWriteEEPROMEvent(uartEventArgs e)
        {
            if (TxWriteEEPROMEvent != null)
            {
                TxWriteEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// Receive Write EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onRxWriteEEPROMEvent(uartEventArgs e)
        {
            if (RxWriteEEPROMEvent != null)
            {
                RxWriteEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// End of Write EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndWriteEEPROMEvent(uartEventArgs e)
        {
            if (endWriteEEPROMEvent != null)
            {
                endWriteEEPROMEvent(this, e);
            }
        }

        /// <summary>
        /// Start Verify Write EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onStartWriteEEPROMVerifyEvent(uartEventArgs e)
        {
            if (startWriteEEPROMVerifyEvent != null)
            {
                startWriteEEPROMVerifyEvent(this, e);
            }
        }

        /// <summary>
        /// Verify Write EEPROM Error event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onWriteEEPROMErrorEvent(uartEventArgs e)
        {
            if (writeEEPROMErrorEvent != null)
            {
                writeEEPROMErrorEvent(this, e);
            }
        }

        /// <summary>
        /// End of Verify Write EEPROM event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndWriteEEPROMVerifyEvent(uartEventArgs e)
        {
            if (endWriteEEPROMVerifyEvent != null)
            {
                endWriteEEPROMVerifyEvent(this, e);
            }
        }

        /// <summary>
        /// Add line to memory list event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onAddMemListLineEvent(uartEventArgs e)
        {
            if (addMemListLineEvent != null)
            {
                addMemListLineEvent(this, e);
            }
        }

        /// <summary>
        /// Add line to hex file parse list event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onAddHexParseLineEvent(uartEventArgs e)
        {
            if (addHexParseLineEvent != null)
            {
                addHexParseLineEvent(this, e);
            }
        }

        /// <summary>
        /// Add line to hex file list event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onAddHexLineEvent(uartEventArgs e)
        {
            if (addHexLineEvent != null)
            {
                addHexLineEvent(this, e);
            }
        }

        /// <summary>
        /// Notifie Config1 value of Hex File event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onHexConfig1Event(uartEventArgs e)
        {
            if (hexConfig1Event != null)
            {
                hexConfig1Event(this, e);
            }
        }

        /// <summary>
        /// Notifie Config2 value of Hex File event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onHexConfig2Event(uartEventArgs e)
        {
            if (hexConfig2Event != null)
            {
                hexConfig2Event(this, e);
            }
        }

        /// <summary>
        /// End of Hex File event thrower
        /// </summary>
        /// <param name="e"></param>
        protected virtual void onEndHexImportEvent(uartEventArgs e)
        {
            if (endHexImportEvent != null)
            {
                endHexImportEvent(this, e);
            }
        }

        /// <summary>
        /// Notifie User High Address value of Hex File event thrower
        /// </summary>
        /// <param name="e"></param>
        private void onUserHighAddrEvent(uartEventArgs e)
        {
            if (userHighAddrEvent != null)
            {
                userHighAddrEvent(this, e);
            }
        }

        /// <summary>
        /// User Application Boundary Error event thrower
        /// </summary>
        /// <param name="e"></param>
        private void onUserAppBoundaryErrorEvent(uartEventArgs e)
        {
            if (userAppBoundaryErrorEvent != null)
            {
                userAppBoundaryErrorEvent(this, e);
            }
        }

        /// <summary>
        /// Run User Application event thrower
        /// </summary>
        /// <param name="e"></param>
        private void onRunUserAppEvent(uartEventArgs e)
        {
            if (runUserAppEvent != null)
            {
                runUserAppEvent(this, e);
            }
        }

        // Event RX Data
        private void comPort_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            // If the com port has been closed, do nothing
            if (!comPort.IsOpen) return;

            //---------------------------------------------------------------
            // PIC send all messages ended by 0x0A
            //---------------------------------------------------------------
            byte[] data = new byte[comPort.BytesToRead]; //fix count byte
            comPort.Read(data, 0, data.Length);
            data.ToList().ForEach(b => receivedData.Enqueue(b));
            //Log(LogMsgType.Incoming, "RXData" + ": " + BitConverter.ToString(data) + "\n");
            //Log(LogMsgType.Incoming, "ProcessData count" + ": " + receivedData.Count + "\n");

            // Determine if we have a Identified Frame in the queue
            if (data.Length > 0)
            {
                if (!(receivedData.First() == LOGIN_TO_DEVICE ||
                    receivedData.First() == WRITE_TO_PROG ||
                    receivedData.First() == WRITE_TO_EEPROM ||
                    receivedData.First() == READ_PROG ||
                    receivedData.First() == READ_EEPROM ||
                    receivedData.First() == ERASE_PROG ||
                    receivedData.First() == CHECKSUM_ERROR))
                {
                    //unIdentified Frame
                    onRxFrameErrorEvent(GetRxEventArg());
                    
                }
                // Parse Received Frame Signature
                else if (receivedData.First() == LOGIN_TO_DEVICE && receivedData.Count == 10)
                {
                    //{ 0x70, 0x0F, 0x26, 0x00, 0xDE, 0x00, 0x04, 0x00, 0x00, 0x0A }
                    //  Id..  Root Addr.  Size......  Offset....  Version...  LF
                    RxLogin();
                }
                else if (receivedData.First() == WRITE_TO_PROG && receivedData.Count == 6)
                {
                    RxWriteMemory();
                }
                else if (receivedData.First() == WRITE_TO_EEPROM && receivedData.Count == 6)
                {
                    RxWriteEEPROM();
                }
                else if (receivedData.First() == CHECKSUM_ERROR && receivedData.Count == 2)
                {
                    ChkSumError();
                }
                else if (receivedData.First() == ERASE_PROG && receivedData.Count == 6)
                {
                    RxEraseMemory();
                }
                else if (receivedData.First() == READ_PROG && receivedData.Count == 6)
                {
                    RxReadMemory();
                }
                else if (receivedData.First() == READ_EEPROM && receivedData.Count == 6)
                {
                    RxReadEEPROM();
                }
            }
        }

        public void Parity_LoadList()
        {
            onParityListEvent(new uartEventArgs(Enum.GetNames(typeof(Parity))));
        }

        public void StopBits_LoadList()
        {
            onStopBitsListEvent(new uartEventArgs(Enum.GetNames(typeof(StopBits))));
        }

        public void PortName_LoadList()
        {
            onPortNameListEvent(new uartEventArgs(OrderedPortNames()));
        }

        public void Device_LoadNameList()
        {
            uartEventArgs args = new uartEventArgs();
            args.Obj = Enum.GetNames(typeof(DeviceName));
            onDeviceNameListEvent(args);
        }

        public void Device_LoadProperties(string name)
        {
            onDevicePropertiesEvent(GetDevicePropertyEventArg(name));
        }

        private uartEventArgs GetRxEventArg()
        {
            uartEventArgs args = new uartEventArgs();
            args.Buffer = new byte[receivedData.Count];
            receivedData.CopyTo(args.Buffer, 0);
            receivedData.Clear();
            return args;
        }

        private string[] OrderedPortNames()
        {
            // Just a placeholder for a successful parsing of a string to an integer
            int num;

            // Order the serial port names in numberic order (if possible)
            return SerialPort.GetPortNames().OrderBy(a => a.Length > 3 && int.TryParse(a.Substring(3), out num) ? num : 0).ToArray();
        }

        internal void RefreshComPortList(IEnumerable<string> previousPortNames, object selectedItem)
        {
            // List of com port names has changed since last checked?
            string selected = RefreshComPortList(previousPortNames, selectedItem as string);

            // If there was an update, then update the control showing the user the list of port names
            if (!String.IsNullOrEmpty(selected))
            {
                //(obj).Items.Clear();
                //(obj).Items.AddRange(OrderedPortNames());
                //(obj).SelectedItem = selected;
                PortName_LoadList();
            }
        }

        private string RefreshComPortList(IEnumerable<string> PreviousPortNames, string CurrentSelection)
        {
            // Create a new return report to populate
            string selected = null;

            // Retrieve the list of ports currently mounted by the operating system (sorted by name)
            string[] ports = SerialPort.GetPortNames();

            // First determain if there was a change (any additions or removals)
            bool updated = PreviousPortNames.Except(ports).Count() > 0 || ports.Except(PreviousPortNames).Count() > 0;

            // If there was a change, then select an appropriate default port
            if (updated)
            {
                // Use the correctly ordered set of port names
                ports = OrderedPortNames();

                // Find newest port if one or more were added
                string newest = SerialPort.GetPortNames().Except(PreviousPortNames).OrderBy(a => a).LastOrDefault();

                // If the port was already open... (see logic notes and reasoning in Notes.txt)
                if (comPort.IsOpen)
                {
                    if (ports.Contains(CurrentSelection)) selected = CurrentSelection;
                    else if (!String.IsNullOrEmpty(newest)) selected = newest;
                    else selected = ports.LastOrDefault();
                }
                else
                {
                    if (!String.IsNullOrEmpty(newest)) selected = newest;
                    else if (ports.Contains(CurrentSelection)) selected = CurrentSelection;
                    else selected = ports.LastOrDefault();
                }
            }

            // If there was a change to the port list, return the recommended default selection
            return selected;
        }

        private uartEventArgs GetDevicePropertyEventArg(string name)
        {
            uartEventArgs args = new uartEventArgs();
            args.Dev = new Device(name, GetPropertyVal(name + "_ID"), GetPropertyVal(name + "_ProgMemSize"),
                GetPropertyVal(name + "_EEPROMsize"), GetPropertyVal(name + "_EEPROMstartAddr"),
                GetPropertyVal(name + "_Config1Addr"), GetPropertyVal(name + "_Config2Addr"));

            config1Addr = args.Dev.Config1Addr.HexToInt();
            config2Addr = args.Dev.Config2Addr.HexToInt();
            EEPROMstartAddr = args.Dev.EEPROMstartAddr.HexToInt();
            EEPROMsize = args.Dev.EEPROMsize.HexToInt();

            // Create Word array and populate
            progMem = new Memory(args.Dev.ProgMemSize.HexToInt()); // default=0x3FFF
            dataEEPROM = new Memory(args.Dev.EEPROMsize.HexToInt(), new Word(0x00, 0xFF), false);
            CopyProg = new Word[args.Dev.ProgMemSize.HexToInt()];
            CopyData = new Word[args.Dev.EEPROMsize.HexToInt()];

            return args;
        }

        private string GetPropertyVal(string name)
        {
            string val = "";
            foreach (PropertyInfo info in settings.GetType().GetProperties())
            {
                if (name.Equals(info.Name))
                {
                    val = (string)info.GetValue(settings, null);
                    break;
                }

            }
            return val;
        }

        #region OpenPort
        public void OpenPort(string BaudRate, string DataBits, string StopBits, string Parity, string PortName)
        {
            try
            {
                //first check if the port is already open
                //if its open then close it
                if (comPort.IsOpen == true) comPort.Close();

                //set the properties of our SerialPort Object
                comPort.BaudRate = int.Parse(BaudRate);    //BaudRate
                comPort.DataBits = int.Parse(DataBits);    //DataBits
                comPort.StopBits = (StopBits)Enum.Parse(typeof(StopBits), StopBits);   //StopBits
                comPort.Parity = (Parity)Enum.Parse(typeof(Parity), Parity);   //Parity
                comPort.PortName = PortName;   //PortName
                //now open the port
                comPort.Open();
                onPortOpenEvent(new uartEventArgs()); // Throw an event
                PortOpen = true;
            }
            catch (Exception ex)
            {
                onPortOpenErrorEvent(new uartEventArgs()); // Throw an event
                PortOpen = false;
            }
        }
        #endregion

        #region TxLogin
        public void TxLogin()
        {
            // Tx Frame = LOGIN
            byte[] TxFrame = { uartCommunication.LOGIN_TO_DEVICE };
            onTxLoginEvent(new uartEventArgs(TxFrame));
            // Send the binary data out the port
            TxData(TxFrame);
        }
        #endregion

        #region RxLogin
        public void RxLogin()
        {
            //{ 0x70, 0x0F, 0x26, 0x00, 0xDE, 0x00, 0x04, 0x00, 0x00, 0x0A }
            //  Id..  Root Addr.  Size......  Offset....  Version...  LF
            uartEventArgs args = GetRxEventArg();
            
            onRxLoginEvent(args);
            Login = true;

            BLoaderRootAddr = Int32.Parse(string.Format("{0}", (args.Buffer[1] * 16 * 16) + args.Buffer[2]));
            BLoaderSize = Int32.Parse(string.Format("{0}", (args.Buffer[3] * 16 * 16) + args.Buffer[4]));
            BLoaderOffset = Int32.Parse(string.Format("{0}", (args.Buffer[5] * 16 * 16) + args.Buffer[6]));

            Login = true;

            DownLoadBLoader();
        }
        #endregion

        #region Read Memory

        private void DownLoadBLoader()
        {

            // Bootloader consists of two disjoint memory sets
            // Reading of sets is planned into ReadPlan table
            progMem.ReadPlan = new MemInterval[]
                {
                    new MemInterval(0,BLoaderOffset-1), //offset
                    new MemInterval(BLoaderRootAddr , //Root
                    BLoaderRootAddr + BLoaderSize - 1) //Root+Size
                    
                };
            progMem.RPlanIndex = 0;
            Verify = false;
            StartReadMemory();
        }

        public void DownLoadMemory()
        {
            // Reading of sets is planned into ReadPlan table
            progMem.ReadPlan = new MemInterval[]
                {
                    new MemInterval(0, BLoaderRootAddr + BLoaderSize - 1) //Root+Size
                };
            progMem.RPlanIndex = 0;
            Verify = false;
            Hex = false;
            StartReadMemory();
        }

        private void StartReadMemory()
        {
            uartEventArgs args = new uartEventArgs();
            args.Min = progMem.ReadPlan[progMem.RPlanIndex].Min;
            args.Max = progMem.ReadPlan[progMem.RPlanIndex].Max;
            args.Step = 1;
            onStartReadMemEvent(args); // Throw an event
            TxReadMemory(progMem.ReadPlan[progMem.RPlanIndex].Min - 1);
        }

        private void TxReadMemory(int index)
        {
            // Next address
            progMem.Index = index + 1;
            // addr[0] = low, addr[1] = high
            byte[] addr = BitConverter.GetBytes(progMem.Index);
            byte[] TxFrame = { uartCommunication.READ_PROG, addr[1], addr[0] };

            Read = false;

            onTxReadMemEvent(new uartEventArgs(TxFrame));
            TxData(TxFrame);
        }

        void RxReadMemory()
        {
            //{ 0x08, 0xha, 0xla, 0xhv, 0xlv, 0x0A }
            //  Id..  Mem. Addr.  Value...... LF
            uartEventArgs args = GetRxEventArg();
            // convert Address to table index
            byte[] addr = new Byte[2];
            addr[0] = args.Buffer[1];
            addr[1] = args.Buffer[2];
            int i = (addr[0] * 16 * 16) + addr[1];

            // store data into table
            progMem.Table[i].high = args.Buffer[3];
            progMem.Table[i].low = args.Buffer[4];

            onRxReadMemEvent(args);

            //Read = true;
            // Read Memory interval from Min to Max
            if (progMem.Index < progMem.ReadPlan[progMem.RPlanIndex].Max)
                TxReadMemory(progMem.Index);
            else
            {
                // End of Memory interval
                 progMem.RPlanIndex++;

                // The Next Memory interval, if there is one, is read
                if (progMem.RPlanIndex < progMem.ReadPlan.Length)
                {
                    StartReadMemory();
                }
                else
                {
                    onEndReadMemEvent(new uartEventArgs()); // Throw an event

                    if (Verify)
                    {
                        WriteMemoryVerify();
                    }
                }
            }
        }
        #endregion

        #region TxData
        /// <summary>
        /// Transmit a Frame to PIC
        /// </summary>
        public void TxData(byte[] buffer)
        {
            comPort.Write(buffer, 0, buffer.Length);
        }
        #endregion

        internal void ViewProgMem(bool protectedArea)
        {
            if (protectedArea)
                DisplayMem(progMem, BLoaderRootAddr,BLoaderRootAddr + BLoaderSize - 1, BLoaderOffset);
            else DisplayMem(progMem, 0, 0, 0);
        }

        public void ViewProgMem(int x, int y, int z)
        {
            DisplayMem(progMem, x, y, z);
        }

        public void ViewEEPROM()
        {
            DisplayMem(dataEEPROM, 0, 0, 0);
        }

        public void ViewEEPROM(int x, int y, int z)
        {
            DisplayMem(dataEEPROM, x, y, z);
        }

        public void DisplayMem(Memory m, int x = 0, int y = 0, int z = 0) //x, y and z optional
        {
 
            bool EEPROM = (m != progMem);
            Word[] w8 = new Word[8];
            for (int i = 0; i < m.Table.Length; i = i + 8)
            {
                uartEventArgs args = new uartEventArgs();
                args.Str = new string[10];
                args.EEPROM = EEPROM;

                Array.Copy(m.Table, i, w8, 0, 8);
                string[] asciiCol = new string[(progMem.TwoBytes ? 16 : 8)];
                args.Str[0] = string.Format("{0,4:X4}", i);
 
                for (int j = 0; j < w8.Length; j++)
                {

                    if (m.TwoBytes)
                    {
                        args.Str[j + 1] = w8[j].WordToString;

                        if (w8[j].High > 31 && w8[j].High < 128)
                            asciiCol[j * 2] = ((char)w8[j].High).ToString();
                        else
                            asciiCol[j * 2] = ".";
                        if (w8[j].Low > 31 && w8[j].Low < 128)
                            asciiCol[j * 2 + 1] = ((char)w8[j].Low).ToString();
                        else
                            asciiCol[j * 2 + 1] = ".";
                    }
                    else
                    {
                        args.Str[j + 1] = w8[j].WordToString.Substring(2);

                        if (w8[j].Low > 31 && w8[j].Low < 128)
                            asciiCol[j] = ((char)w8[j].Low).ToString();
                        else
                            asciiCol[j] = ".";
                    }

                    args.ProtectedArea = false;
                    if (i + j < z)
                    {
                        args.ProtectedArea = true;
                    }

                    else if (y != 0 && i + j >= x && i + j <= y)
                    {
                        args.ProtectedArea = true;
                    }
               }
                args.Str[9] = string.Join("", asciiCol);
                onAddMemListLineEvent(args);
            }
        }

        public void ImportHexFile(string hexFile)
        {
            int counter = 0;
            Int32 ptr = 0;
            string line;
            Int32 n = 0;
            Int32 addr = 0;
            byte[] buffer;

            // Read the file and display it line by line.
            System.IO.StreamReader file =
                new System.IO.StreamReader(@hexFile, true);

            while ((line = file.ReadLine()) != null)
            {

                buffer = System.Text.ASCIIEncoding.Default.GetBytes(line);

                uartEventArgs a = new uartEventArgs();
                a.Str = new string[1];
                a.Str[0] = line + "\n";
                a.Buffer = new byte[buffer.Length];
                Array.Copy(buffer, a.Buffer, buffer.Length);
                onAddHexLineEvent(a);

                // Bytes count
                n = ("" + (char)buffer[1] + (char)buffer[2]).HexToInt();

                addr = ("" + (char)buffer[3] + (char)buffer[4] + (char)buffer[5] + (char)buffer[6]).HexToInt();
                if (buffer[8] == '0') //Type Enr=00
                {
                    for (int i = 0; i < n; i = i + 2)
                    {

                        // OpCode Address
                        ptr = (addr + i) / 2;
                        // OpCode
                        string opCode = "" + (char)(buffer[11 + 2 * i]) + (char)(buffer[12 + 2 * i]) + (char)(buffer[9 + 2 * i]) + (char)(buffer[10 + 2 * i]);
                        byte[] opCodeH = BitConverter.GetBytes(("" + (char)(buffer[11 + 2 * i]) + (char)(buffer[12 + 2 * i])).HexToInt());
                        byte[] opCodeL = BitConverter.GetBytes(("" + (char)(buffer[9 + 2 * i]) + (char)(buffer[10 + 2 * i])).HexToInt());

                        uartEventArgs args = new uartEventArgs();
                        args.Str = new string[3];

                        args.Str[0] = string.Format("{0,4:D4}", ++counter);

                        string ptrHexString = ("0000" + (ptr).ToString("X"));
                        args.Str[1] = (ptrHexString.Substring(ptrHexString.Length - 4));

                        args.Str[2]= opCode;
                        // Event add a line
                        onAddHexParseLineEvent(args);

                        if (ptr < progMem.Table.Length)
                        {
                            progMem.Table[ptr].high = opCodeH[0];
                            progMem.Table[ptr].low = opCodeL[0];
                            //UserHighAddrDTextBox.Text = string.Format("{0:D4}", ptr);
                            string str = string.Format("{0,4:X4}", ptr);
                            //UserHighAddrTextBox.Text = str.Substring(0, 2) + "-" + str.Substring(2);
                            onUserHighAddrEvent(new uartEventArgs( ptr));
                            UserHighAddr = ptr;
                        }
                        //CONFIG
                        //config1TextBox.Text = opCode;
                        if (ptr == config1Addr) onHexConfig1Event(new uartEventArgs(new string[] { opCode }));
                        //config2TextBox.Text = opCode;
                        if (ptr == config2Addr) onHexConfig2Event(new uartEventArgs(new string[] { opCode }));
                        //EEPROM
                        if (ptr >= EEPROMstartAddr)
                        {
                            dataEEPROM.Table[ptr - EEPROMstartAddr].high = opCodeH[0];
                            dataEEPROM.Table[ptr - EEPROMstartAddr].low = opCodeL[0];
                        }
                    }
                }
            }
            file.Close();
            Hex = true;
            onEndHexImportEvent(new uartEventArgs());

            // Boundary control
            if (Login)
            {
                if (BLoaderRootAddr < UserHighAddr)
                {
                    HexError = true;
                    // Displays a message box boundary error
                    onUserAppBoundaryErrorEvent(new uartEventArgs());
                }
                else HexError = false;
            }
        }

        public void UpLoadUserApp()
        {
            // 16F88 requires explicit 32 words block erase and then 4 words block write
            // User application consists of one memory set of n*32 blocks
            // Erasing and Writing of set are planned into WritePlan table
            int max = (UserHighAddr % 32 == 0) ? UserHighAddr: ((UserHighAddr / 32) + 1) * 32;
            if (BLoaderRootAddr < UserHighAddr)
            {
                HexError = true;
                // Displays a message box boundary error
                onUserAppBoundaryErrorEvent(new uartEventArgs());
            }
            else 
            {
                HexError = false;
                progMem.WritePlan = new MemInterval[]
                {
                    new MemInterval(BLoaderOffset ,max - 1)
                };
                progMem.WPlanIndex = 0;
                //StartEraseMemory();
                StartEraseMemory();
            }
        }

        #region Erase Memory

        private void StartEraseMemory()
        {
            uartEventArgs args = new uartEventArgs();
            args.Min = progMem.WritePlan[progMem.WPlanIndex].Min;
            args.Max = progMem.WritePlan[progMem.WPlanIndex].Max;
            args.Step = 1;
            onStartEraseMemEvent(args); // Throw an event
            TxEraseMemory(progMem.WritePlan[progMem.WPlanIndex].Min - 32);
        }

        private void TxEraseMemory(int index)
        {
            // Next address
            progMem.Index = index + 32;

            // addr[0] = low, addr[1] = high
            byte[] addr = BitConverter.GetBytes(progMem.Index);

            // Set Frame without neither ID nor cheksum 
            byte[] TxFrame = { uartCommunication.ERASE_PROG, addr[1], addr[0], 
                                 //progMem.Table[progMem.Index].High, progMem.Table[progMem.Index].Low, 0x00 };
                                 };

            Erase = false;

            onTxEraseMemEvent(new uartEventArgs(TxFrame));
            // Send the binary data via the port
            TxData(TxFrame);
        }

        private void RxEraseMemory()
        {
            //{ 0x20, 0xha, 0xla, 0x00, 0x00, 0x0A }
            //  Id..  Mem. Addr.  Value...... LF
            onRxEraseMemEvent(GetRxEventArg());

            //Erase = true;
            // Erase Memory interval from Min to Max
            if (progMem.Index < progMem.WritePlan[progMem.WPlanIndex].Max - 32)
                TxEraseMemory(progMem.Index);
            else
            {
                // End of Memory interval
                onEndEraseMemEvent(new uartEventArgs()); // Throw an event
                progMem.WPlanIndex++;

                // The Next Memory interval, if there is one, is erased
                if (progMem.WPlanIndex < progMem.WritePlan.Length)
                {
                    StartEraseMemory();
                }
                else
                {
                    // At end of Erase, Start Write
                    progMem.WPlanIndex = 0;
                    StartWriteMemory();
                }
            }
        }

        private void StartWriteMemory()
        {
            uartEventArgs args = new uartEventArgs();
            args.Min = progMem.WritePlan[progMem.WPlanIndex].Min;
            args.Max = progMem.WritePlan[progMem.WPlanIndex].Max;
            args.Step = 1;
            onStartWriteMemEvent(args); // Throw an event
            TxWriteMemory(progMem.WritePlan[progMem.WPlanIndex].Min - 1);
        }

        private void TxWriteMemory(int index)
        {
            // Next address
            progMem.Index = index + 1;

            // addr[0] = low, addr[1] = high
            byte[] addr = BitConverter.GetBytes(progMem.Index);

            // Set Frame without neither ID nor cheksum 
            byte[] TxFrame = { 0x00, addr[1], addr[0],
                                 progMem.Table[progMem.Index].High, progMem.Table[progMem.Index].Low, 0x00 };
            //0x00, 0x00, 0x00 };

            // compute checksum
            int checksum = 0;
            foreach (byte b in TxFrame) checksum += b;
            checksum &= 0xff;

            TxFrame[TxFrame.Length - 1] = (byte)checksum;
            TxFrame[0] = uartCommunication.WRITE_TO_PROG;

            onTxWriteMemEvent(new uartEventArgs(TxFrame));

            //Hold frame for checksumerror
            LastTx = new byte[TxFrame.Length];
            Array.Copy(TxFrame, LastTx, TxFrame.Length);

            Write = false;
            // Send the binary data via the port
            TxData(TxFrame);
        }

        private void RxWriteMemory()
        {
            //{ 0x02, 0xha, 0xla, 0xhv, 0xlv, 0x0A }
            //  Id..  Mem. Addr.  Value...... LF
            onRxWriteMemEvent(GetRxEventArg());

            //Write = true;
            // 1) Read Bootloader reset vector
            if (progMem.Index < progMem.WritePlan[progMem.WPlanIndex].Max)
                TxWriteMemory(progMem.Index);
            else
            {
                // End of Memory interval
                onEndWriteMemEvent(new uartEventArgs());
                progMem.WPlanIndex++;

                // The Next Memory interval, if there is one, is writen
                if (progMem.WPlanIndex < progMem.WritePlan.Length)
                {
                    StartWriteMemory();
                }
                else
                {
                    StartWriteMemoryVerify();
                }
            }
        }

        private void StartWriteMemoryVerify()
        {
            // At end of Write
            // Start Verify
            Array.Copy(progMem.Table, CopyProg, CopyProg.Length);
            // Re-Reading plan = Write plan
            progMem.ReadPlan = new MemInterval[progMem.WritePlan.Length];
            Array.Copy(progMem.WritePlan, progMem.ReadPlan, progMem.WritePlan.Length);
            progMem.RPlanIndex = 0;

            Verify = true;
            onStartWriteMemVerifyEvent(new uartEventArgs());

            StartReadMemory();
        }

        private void WriteMemoryVerify()
        {

            if (Verify)
            {
                int errorCount = 0;
                for (int j = 0; j < CopyProg.Length; j++)
                {
                    if (!(progMem.Table[j].High == CopyProg[j].High && progMem.Table[j].Low == CopyProg[j].Low))
                    {
                        onWriteMemErrorEvent(new uartEventArgs (j, new string[] { progMem.Table[j].WordToString, CopyProg[j].WordToString }));
                        errorCount++;
                    }
                }

                Verify = false;

                onEndWriteMemVerifyEvent(new uartEventArgs());
            }
        }

        private void ChkSumError()
        {
             onChkSumErrorEvent(new uartEventArgs(LastTx)); // Throw an event
        }

        #endregion

        #region Read EEPROM
        public void DownLoadEEPROM()
        {
            dataEEPROM.ReadPlan = new MemInterval[] { new MemInterval(0, EEPROMsize - 1) };
            dataEEPROM.RPlanIndex = 0;
            Verify = false;
            Hex = false;
            StartReadEEPROM();
        }

        private void StartReadEEPROM()
        {
            uartEventArgs args = new uartEventArgs();
            args.Min = dataEEPROM.ReadPlan[dataEEPROM.RPlanIndex].Min;
            args.Max = dataEEPROM.ReadPlan[dataEEPROM.RPlanIndex].Max;
            args.Step = 1;
            onStartReadEEPROMEvent(args); // Throw an event
            TxReadEEPROM(dataEEPROM.ReadPlan[dataEEPROM.RPlanIndex].Min - 1);
        }

        private void TxReadEEPROM(int index)
        {
            // Next address
            dataEEPROM.Index = index + 1;
            // addr[0] = low, addr[1] = high
            byte[] addr = BitConverter.GetBytes(dataEEPROM.Index);
            byte[] TxFrame = { uartCommunication.READ_EEPROM, addr[1], addr[0] };

            Read = false;

            onTxReadEEPROMEvent(new uartEventArgs(TxFrame));
            TxData(TxFrame);
        }

        void RxReadEEPROM()
        {
            //{ 0x10, 0xha, 0xla, 0xhv, 0xlv, 0x0A }
            //  Id..  Mem. Addr.  Value...... LF
            uartEventArgs args = GetRxEventArg();
            // convert Address to table index
            byte[] addr = new Byte[2];
            addr[0] = args.Buffer[1];
            addr[1] = args.Buffer[2];
            int i = (addr[0] * 16 * 16) + addr[1];

            // store data into table
            dataEEPROM.Table[i].high = args.Buffer[3];
            dataEEPROM.Table[i].low = args.Buffer[4];

            onRxReadEEPROMEvent(args);

            //Read = true;
            // Read Memory interval from Min to Max
            if (dataEEPROM.Index < dataEEPROM.ReadPlan[dataEEPROM.RPlanIndex].Max)
                TxReadEEPROM(dataEEPROM.Index);
            else
            {
                // End of Memory interval
                dataEEPROM.RPlanIndex++;

                // The Next Memory interval, if there is one, is read
                if (dataEEPROM.RPlanIndex < dataEEPROM.ReadPlan.Length)
                {
                    StartReadEEPROM();
                }
                else
                {
                    onEndReadEEPROMEvent(new uartEventArgs()); // Throw an event

                    if (Verify)
                    {
                        WriteEEPROMVerify();
                    }
                }
            }
        }

        public void UploadUserData()
        {
            dataEEPROM.WritePlan = new MemInterval[] { new MemInterval(0, EEPROMsize - 1) };
            dataEEPROM.WPlanIndex = 0;

            StartWriteEEPROM();
        }

        private void StartWriteEEPROM()
        {
            uartEventArgs args = new uartEventArgs();
            args.Min = dataEEPROM.WritePlan[dataEEPROM.WPlanIndex].Min;
            args.Max = dataEEPROM.WritePlan[dataEEPROM.WPlanIndex].Max;
            args.Step = 1;
            onStartWriteEEPROMEvent(args); // Throw an event
            TxWriteEEPROM(dataEEPROM.WritePlan[dataEEPROM.WPlanIndex].Min - 1);
        }

        private void TxWriteEEPROM(int index)
        {
            // Next address
            dataEEPROM.Index = index + 1;

            // addr[0] = low, addr[1] = high
            byte[] addr = BitConverter.GetBytes(dataEEPROM.Index);

            // Set Frame without neither ID nor cheksum 
            byte[] TxFrame = { 0x00, addr[1], addr[0],
                                 dataEEPROM.Table[dataEEPROM.Index].High, dataEEPROM.Table[dataEEPROM.Index].Low, 0x00 };

            // compute checksum
            int checksum = 0;
            foreach (byte b in TxFrame) checksum += b;
            checksum &= 0xff;

            TxFrame[TxFrame.Length - 1] = (byte)checksum;
            TxFrame[0] = uartCommunication.WRITE_TO_EEPROM;

            onTxWriteEEPROMEvent(new uartEventArgs(TxFrame));

            LastTx = new byte[TxFrame.Length];
            Array.Copy(TxFrame, LastTx, TxFrame.Length);

            Write = false;
            // Send the binary data via the port
            TxData(TxFrame);
        }

        private void RxWriteEEPROM()
        {
            //{ 0x04, 0xha, 0xla, 0xhv, 0xlv, 0x0A }
            //  Id..  Mem. Addr.  Value...... LF
            onRxWriteEEPROMEvent(GetRxEventArg());

            //Write = true;
            // 1) Read Bootloader reset vector
            if (dataEEPROM.Index < dataEEPROM.WritePlan[dataEEPROM.WPlanIndex].Max)
                TxWriteEEPROM(dataEEPROM.Index);
            else
            {
                // End of Memory interval
                onEndWriteEEPROMEvent(new uartEventArgs());
                dataEEPROM.WPlanIndex++;

                // The Next Memory interval, if there is one, is writen
                if (dataEEPROM.WPlanIndex < dataEEPROM.WritePlan.Length)
                {
                    StartWriteEEPROM();
                }
                else
                {
                    // At end of Write
                    // Start Verify
                    StartWriteEEPROMVerify();
                }
            }
        }

        private void StartWriteEEPROMVerify()
        {
            Array.Copy(dataEEPROM.Table, CopyData, CopyData.Length);
            // Re-Reading plan = Write plan
            dataEEPROM.ReadPlan = new MemInterval[1];
            Array.Copy(dataEEPROM.WritePlan, dataEEPROM.ReadPlan, 1);
            dataEEPROM.RPlanIndex = 0;

            Verify = true;
            onStartWriteEEPROMVerifyEvent(new uartEventArgs());
            StartReadEEPROM();
        }

        private void WriteEEPROMVerify()
        {
            int errorCount = 0;
            for (int j = 0; j < CopyData.Length; j++)
            {
                if (!(dataEEPROM.Table[j].High == CopyData[j].High && dataEEPROM.Table[j].Low == CopyData[j].Low))
                {
                    onWriteEEPROMErrorEvent(new uartEventArgs(j, new string[] { dataEEPROM.Table[j].WordToString, CopyData[j].WordToString }));
                    errorCount++;
                }
            }
            onEndWriteEEPROMVerifyEvent(new uartEventArgs());

         }

        #endregion

        public void RunUserApp()
        {
            byte[] TxFrame = { uartCommunication.LOGOUT_OF_DEVICE };
            onRunUserAppEvent(new uartEventArgs(TxFrame));
            TxData(TxFrame);
            Login = false;

        }
    }
}
